<?php
/**
 * POST /api/detail.php
 * Returns full site detail for a single .onion address.
 * Public, rate-limited.
 */

require_once __DIR__ . '/../includes/security.php';

Security::requirePost();

$cfg = require __DIR__ . '/../includes/config.php';
if (!Security::checkRate(Security::clientIp(), $cfg['rate_limit'], $cfg['rate_window_sec'])) {
    Security::json(['error' => 'Rate limit exceeded'], 429);
}

$input = json_decode(file_get_contents('php://input'), true) ?: $_POST;
$address = trim($input['address'] ?? '');

if (!preg_match('/^[a-z2-7]{56}\.onion$/', $address)) {
    Security::json(['error' => 'Invalid address'], 400);
}

$db = DB::get();

// Main record
$stmt = $db->prepare("SELECT * FROM onions WHERE address = ?");
$stmt->execute([$address]);
$site = $stmt->fetch();
if (!$site) {
    Security::json(['error' => 'Not found'], 404);
}

// Sub-pages
$stmt = $db->prepare(
    "SELECT page_url, title, source, first_seen, last_seen
     FROM pages WHERE onion_id = ? ORDER BY first_seen DESC LIMIT 100"
);
$stmt->execute([$site['id']]);
$pages = $stmt->fetchAll();

// Uptime history (last 30 days, max 90 entries)
$stmt = $db->prepare(
    "SELECT alive, response_ms, checked_at
     FROM uptime_log
     WHERE address = ? AND checked_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
     ORDER BY checked_at ASC
     LIMIT 90"
);
$stmt->execute([$address]);
$uptime = $stmt->fetchAll();

// Report count
$stmt = $db->prepare("SELECT COUNT(*) FROM reports WHERE address = ? AND status = 'pending'");
$stmt->execute([$address]);
$reportCount = (int) $stmt->fetchColumn();

Security::json([
    'site'    => $site,
    'pages'   => $pages,
    'uptime'  => $uptime,
    'reports' => $reportCount,
]);
