<?php
/**
 * GET /api/feed.php?type=sites|pages
 * Public RSS/Atom feed. No auth, rate limited.
 */

require_once __DIR__ . '/../includes/security.php';

$cfg = require __DIR__ . '/../includes/config.php';
if (!Security::checkRate(Security::clientIp(), $cfg['rate_limit'], $cfg['rate_window_sec'])) {
    http_response_code(429);
    echo 'Rate limited';
    exit;
}

$db = DB::get();
$type = $_GET['type'] ?? 'sites';
$siteName = $cfg['site_name'] ?? 'In Umbra';

header('Content-Type: application/atom+xml; charset=utf-8');
header('X-Content-Type-Options: nosniff');

$items = [];

if ($type === 'pages') {
    $stmt = $db->query(
        "SELECT p.page_url, p.title AS page_title, p.first_seen,
                o.address, o.title AS site_title
         FROM pages p JOIN onions o ON o.id = p.onion_id
         ORDER BY p.first_seen DESC LIMIT 50"
    );
    $items = $stmt->fetchAll();
} else {
    $stmt = $db->query(
        "SELECT address, title, status, cti_type, meta_description, first_seen, last_seen
         FROM onions ORDER BY first_seen DESC LIMIT 50"
    );
    $items = $stmt->fetchAll();
}

$feedTitle = $type === 'pages' ? "$siteName — Newest Pages" : "$siteName — Newest Sites";
// Convert MySQL datetime "2025-01-01 12:00:00" to Atom "2025-01-01T12:00:00Z"
function atomDate($d) { return str_replace(' ', 'T', $d) . 'Z'; }
$updated = $items ? atomDate($items[0]['first_seen'] ?? gmdate('Y-m-d H:i:s')) : gmdate('Y-m-d\TH:i:s\Z');

echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
?>
<feed xmlns="http://www.w3.org/2005/Atom">
  <title><?= htmlspecialchars($feedTitle) ?></title>
  <subtitle>inumbra.com — dark web meta-search</subtitle>
  <updated><?= htmlspecialchars($updated) ?></updated>
  <id>urn:umbra:feed:<?= $type ?></id>
<?php if ($type === 'pages'): ?>
<?php foreach ($items as $i): ?>
  <entry>
    <title><?= htmlspecialchars($i['page_title'] ?: 'Untitled page') ?></title>
    <id>urn:umbra:page:<?= htmlspecialchars(md5($i['page_url'])) ?></id>
    <updated><?= atomDate($i['first_seen']) ?></updated>
    <summary>Page on <?= htmlspecialchars($i['site_title'] ?: $i['address']) ?> (<?= htmlspecialchars($i['address']) ?>)</summary>
    <content type="text"><?= htmlspecialchars($i['page_url']) ?></content>
  </entry>
<?php endforeach; ?>
<?php else: ?>
<?php foreach ($items as $i): ?>
  <entry>
    <title><?= htmlspecialchars($i['title'] ?: $i['address']) ?></title>
    <id>urn:umbra:site:<?= htmlspecialchars($i['address']) ?></id>
    <updated><?= atomDate($i['first_seen']) ?></updated>
    <summary><?= htmlspecialchars(mb_substr($i['meta_description'] ?: $i['cti_type'], 0, 300)) ?></summary>
    <category term="<?= htmlspecialchars($i['status']) ?>"/>
    <?php if ($i['cti_type']): ?><category term="<?= htmlspecialchars($i['cti_type']) ?>"/><?php endif; ?>
    <content type="text"><?= htmlspecialchars($i['address']) ?> [<?= $i['status'] ?>]</content>
  </entry>
<?php endforeach; ?>
<?php endif; ?>
</feed>
