<?php
/**
 * In Umbra — Removal Request API
 * POST: {type, address?, email?, detail, captcha_answer, captcha_token}
 */
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/security.php';
$cfg = require __DIR__ . '/../includes/config.php';
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') { echo json_encode(['error'=>'POST required']); exit; }
$d = json_decode(file_get_contents('php://input'), true);
if (!$d) { echo json_encode(['error'=>'Invalid JSON']); exit; }

// Fix #9: CSRF check
$_rCsrf = $d['_csrf'] ?? '';
$_cCsrf = $_COOKIE['_csrf'] ?? '';
if ($_cCsrf === '' || !hash_equals($_cCsrf, $_rCsrf)) {
    echo json_encode(['error'=>'Invalid request token. Refresh and try again.']); exit;
}

$type   = trim($d['type'] ?? '');
$addr   = trim($d['address'] ?? '');
$email  = trim($d['email'] ?? '');
$detail = trim($d['detail'] ?? '');
$cAns   = intval($d['captcha_answer'] ?? -1);
$cTok   = $d['captcha_token'] ?? '';

// Validate type
$validTypes = ['delist','personal','dmca','other'];
if (!in_array($type, $validTypes)) { echo json_encode(['error'=>'Invalid request type']); exit; }

// Validate address format for delist requests
if ($type === 'delist' && $addr !== '') {
    if (!preg_match('/^[a-z2-7]{56}\.onion$/', strtolower($addr))) {
        echo json_encode(['error'=>'Invalid .onion address (must be 56-character v3 address)']); exit;
    }
    $addr = strtolower($addr);
}

// Validate detail
if (strlen($detail) < 10) { echo json_encode(['error'=>'Please provide more detail (at least 10 characters)']); exit; }
if (strlen($detail) > 2000) $detail = substr($detail, 0, 2000);

// Validate email format (if provided)
if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) { echo json_encode(['error'=>'Invalid email address']); exit; }

// Validate captcha (HMAC)
// Fix #28: Token format: "ts:hmac(answer:ts)"
$secret = 'inumbra-captcha-' . ($cfg['db_name'] ?? 'salt');
$parts = explode(':', $cTok);
if (count($parts) !== 2) { echo json_encode(['error'=>'Invalid captcha token']); exit; }
[$ts, $sig] = $parts;
if (time() - intval($ts) > 300) { echo json_encode(['error'=>'Captcha expired, please reload']); exit; }
$expect = hash_hmac('sha256', "$cAns:$ts", $secret);
if (!hash_equals($expect, $sig)) { echo json_encode(['error'=>'Wrong captcha answer']); exit; }

// Rate limit — max 10 pending removal requests per IP hash
$db = DB::get();
$ipHash = hash('sha256', $_SERVER['REMOTE_ADDR'] ?? '');

try {
    $cnt = $db->prepare("SELECT COUNT(*) FROM removal_requests WHERE ip_hash=? AND status='pending'");
    $cnt->execute([$ipHash]);
    if ((int)$cnt->fetchColumn() >= 10) {
        echo json_encode(['message'=>'Request noted. You have several pending requests — we will review them.']);
        exit;
    }
} catch (\Throwable $e) {}

// Insert
try {
    $st = $db->prepare("INSERT INTO removal_requests (type, address, email, detail, ip_hash, status, created_at) VALUES (?,?,?,?,?,'pending',NOW())");
    $st->execute([$type, $addr, $email, $detail, $ipHash]);
    echo json_encode(['message'=>'Request submitted. We aim to process removal requests within 72 hours.']);
} catch (\Throwable $e) {
    echo json_encode(['error'=>'Failed to submit request. Please try again or email removal@inumbra.com']);
}
