<?php
/**
 * In Umbra — Security helpers
 */

require_once __DIR__ . '/db.php';

class Security {

    /**
     * Check rate limit for a key. Returns true if allowed, false if exceeded.
     * Fix #22: Uses INSERT...ON DUPLICATE KEY UPDATE for atomicity.
     */
    public static function checkRate(string $key, int $limit = 30, int $windowSec = 60): bool {
        $db = DB::get();
        $keyHash = hash('sha256', $key);

        // Atomic upsert: if window expired, reset; otherwise increment
        $db->prepare(
            "INSERT INTO rate_limits (ip_hash, requests, window_start)
             VALUES (?, 1, NOW())
             ON DUPLICATE KEY UPDATE
                requests = IF(window_start < DATE_SUB(NOW(), INTERVAL ? SECOND), 1, requests + 1),
                window_start = IF(window_start < DATE_SUB(NOW(), INTERVAL ? SECOND), NOW(), window_start)"
        )->execute([$keyHash, $windowSec, $windowSec]);

        // Now check the count
        $stmt = $db->prepare("SELECT requests FROM rate_limits WHERE ip_hash = ?");
        $stmt->execute([$keyHash]);
        $row = $stmt->fetch();

        return $row && $row['requests'] <= $limit;
    }

    /**
     * Validate API key from Authorization header.
     */
    public static function validateApiKey(): string|false {
        $header = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['HTTP_X_API_KEY'] ?? '';
        $key = str_replace('Bearer ', '', $header);
        if (strlen($key) < 32) return false;

        $db = DB::get();
        $keyHash = hash('sha256', $key);
        $stmt = $db->prepare(
            "SELECT id, permissions FROM api_keys WHERE key_hash = ? AND active = 1"
        );
        $stmt->execute([$keyHash]);
        $row = $stmt->fetch();

        if (!$row) return false;

        $db->prepare("UPDATE api_keys SET last_used = NOW() WHERE id = ?")
           ->execute([$row['id']]);

        return $row['permissions'];
    }

    /**
     * Fix #5: Get client IP safely.
     * Only trusts X-Forwarded-For from known proxy addresses.
     */
    public static function clientIp(): string {
        $remoteAddr = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';

        $cfg = @include __DIR__ . '/config.php';
        $trusted = $cfg['trusted_proxies'] ?? ['127.0.0.1', '::1'];

        $isTrusted = false;
        foreach ($trusted as $proxy) {
            if (str_contains($proxy, '/')) {
                if (self::ipInCidr($remoteAddr, $proxy)) { $isTrusted = true; break; }
            } else {
                if ($remoteAddr === $proxy) { $isTrusted = true; break; }
            }
        }

        if ($isTrusted) {
            $xff = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? '';
            if ($xff !== '') {
                $ips = array_map('trim', explode(',', $xff));
                $clientIp = $ips[0];
                if (filter_var($clientIp, FILTER_VALIDATE_IP)) {
                    return $clientIp;
                }
            }
            $xri = $_SERVER['HTTP_X_REAL_IP'] ?? '';
            if ($xri !== '' && filter_var($xri, FILTER_VALIDATE_IP)) {
                return $xri;
            }
        }

        return $remoteAddr;
    }

    private static function ipInCidr(string $ip, string $cidr): bool {
        [$subnet, $mask] = explode('/', $cidr, 2);
        $mask = (int)$mask;
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) && filter_var($subnet, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            return (ip2long($ip) & ~((1 << (32 - $mask)) - 1)) === (ip2long($subnet) & ~((1 << (32 - $mask)) - 1));
        }
        return false;
    }

    /**
     * Fix #12: Send standard security headers including CSP.
     */
    public static function securityHeaders(): void {
        header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'; img-src 'self' data:; font-src 'self'; connect-src 'self'; frame-ancestors 'none'");
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: DENY');
        header('Referrer-Policy: strict-origin-when-cross-origin');
    }

    /**
     * Fix #9: Generate CSRF token (session-less, cookie-based).
     */
    public static function csrfToken(): string {
        if (isset($_COOKIE['_csrf'])) {
            return $_COOKIE['_csrf'];
        }
        $token = bin2hex(random_bytes(16));
        setcookie('_csrf', $token, [
            'expires'  => 0,
            'path'     => '/',
            'httponly' => true,
            'samesite' => 'Strict',
            'secure'   => !empty($_SERVER['HTTPS']),
        ]);
        return $token;
    }

    /**
     * Fix #9: Validate CSRF token. Skips for GET (safe/idempotent).
     */
    public static function csrfCheck(): bool {
        if ($_SERVER['REQUEST_METHOD'] === 'GET') return true;
        $cookie = $_COOKIE['_csrf'] ?? '';
        $posted = $_POST['_csrf'] ?? $_GET['_csrf'] ?? '';
        return $cookie !== '' && hash_equals($cookie, $posted);
    }

    /** Send JSON response and exit. */
    public static function json(array $data, int $code = 200): never {
        http_response_code($code);
        header('Content-Type: application/json; charset=utf-8');
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: DENY');
        header('Referrer-Policy: strict-origin-when-cross-origin');
        echo json_encode($data, JSON_UNESCAPED_UNICODE);
        exit;
    }

    /** Require POST method or die. */
    public static function requirePost(): void {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            self::json(['error' => 'Method not allowed'], 405);
        }
    }

    /** Require valid API key or die. */
    public static function requireAuth(): string {
        $perms = self::validateApiKey();
        if ($perms === false) {
            self::json(['error' => 'Unauthorized'], 401);
        }
        return $perms;
    }
}
