<?php
/**
 * In Umbra — Frontend v2
 * Views: home | search | newest | tag | detail | tech | lang | changes |
 *        canaries | mirrors | submit | privacy | terms | removal
 * Features: advanced search, trending, sparklines, reports, RSS, legal pages
 */
require_once __DIR__ . '/../includes/db.php';
$cfg = require __DIR__ . '/../includes/config.php';

// Fix #12: Send CSP and security headers
require_once __DIR__ . '/../includes/security.php';
Security::securityHeaders();

// Fix #9: CSRF token for forms
$csrfToken = Security::csrfToken();

$db = DB::get();
$SN = htmlspecialchars($cfg['site_name']);
$LIM = $cfg['results_per_page'];
$_ri = 0; // Fix #31: Still sequential but used only for unique DOM IDs

// Accept both GET and POST (GET for crawlers/links, POST for forms)
$_R = array_merge($_GET, $_POST);
$V = $_R['v'] ?? '';
$Q = trim($_R['q'] ?? '');
$TAG = trim($_R['tag'] ?? '');
$ADDR = trim($_R['addr'] ?? '');
$PG = max(1, intval($_R['p'] ?? 1));
$OFF = ($PG - 1) * $LIM;

$res = []; $tot = 0; $pgs = 1; $err = '';
$allTags = []; $trend = []; $det = null; $sparks = [];
$PT = $SN;

// ── Helpers ──
function boolQ(string $r): string {
    $c = preg_replace('/[^\p{L}\p{N}\s.\-]/u', '', $r);
    $c = trim($c); if ($c === '') return '';
    return implode(' ', array_map(fn($w) => '+' . $w . '*', preg_split('/\s+/', $c)));
}
function parseAdv(string $raw): array {
    $f = ['status'=>'','cti'=>'','tech'=>'','lang'=>'','after'=>'','before'=>''];
    $kw = $raw;
    foreach (array_keys($f) as $k) {
        if (preg_match('/'.$k.':(\S+)/i', $kw, $m)) { $f[$k] = $m[1]; $kw = str_replace($m[0], '', $kw); }
    }
    return [trim(preg_replace('/\s+/', ' ', $kw)), $f];
}
function safeTerm(string $t): bool {
    // Word-boundary matching to avoid false blocks on "sextant", "teenage", "assignment" etc.
    // Short terms (<=3 chars) use exact match; longer terms use word boundary regex
    $exact = ['cp','xxx','fap'];  // Only block these as exact word matches
    $boundary = ['child','pedo','pthc','loli','shota','jailbait','underage','preteen',
            'infant','toddler','molest','rape','incest','porn','sex','nude','naked',
            'hentai','nsfw','bdsm','fetish','escort','webcam','camgirl','onlyfans',
            'milf','teen','nymphet','kiddy','csam'];
    $words = preg_split('/\s+/', strtolower(trim($t)));
    foreach ($words as $w) {
        if (in_array($w, $exact)) return false;
        foreach ($boundary as $b) {
            if ($w === $b) return false;  // Exact word match only
        }
    }
    return true;
}
function spark(array $pts, bool $lg = false): string {
    if (!$pts) return '';
    $h = $lg ? 30 : 14; $w = $lg ? 6 : 3; $cls = $lg ? 'spark-lg' : 'spark';
    $s = '<span class="'.$cls.'">';
    foreach (array_slice($pts, -($lg ? 60 : 30)) as $p) {
        $c = $p ? 'var(--on)' : 'var(--w)';
        $ht = $p ? $h : max(3, intval($h * 0.3));
        $s .= "<i style=\"height:{$ht}px;background:$c;width:{$w}px\"></i>";
    }
    return $s . '</span>';
}

// ════════ SEARCH ════════
if ($V === 'search' && $Q !== '') {
    // Input length limit
    if (mb_strlen($Q) > 300) $Q = mb_substr($Q, 0, 300);
    [$kw, $flt] = parseAdv($Q);
    $PT = htmlspecialchars($Q) . " — $SN";
    $bt = boolQ($kw);
    // Log trending
    if ($kw !== '' && safeTerm($kw) && mb_strlen($kw) >= 2 && mb_strlen($kw) <= 60) {
        try { $db->prepare("INSERT INTO trending_terms (term,hits,last_seen) VALUES (?,1,NOW()) ON DUPLICATE KEY UPDATE hits=hits+1,last_seen=NOW()")->execute([mb_strtolower($kw)]); } catch (\Throwable $e) {}
    }
    // Ensure we have at least one filter or keyword
    $hasAnyFilter = ($bt !== '' || $flt['status'] || $flt['cti'] || $flt['tech'] || $flt['lang'] || $flt['after'] || $flt['before']);
    if (!$hasAnyFilter) {
        $err = 'Enter a search term or filter.';
    } else {
    try {
        $w = []; $p = [];
        if ($bt !== '') { $w[] = "MATCH(title,meta_description,meta_keywords,tech_stack,tags,notes,cti_name,cti_type,server_software) AGAINST(? IN BOOLEAN MODE)"; $p[] = $bt; }
        if ($flt['status']) { $w[] = "status=?"; $p[] = $flt['status']; }
        if ($flt['cti'])    { $w[] = "cti_type LIKE ?"; $p[] = '%'.$flt['cti'].'%'; }
        if ($flt['tech'])   { $w[] = "tech_stack LIKE ?"; $p[] = '%'.$flt['tech'].'%'; }
        if ($flt['lang'])   { $w[] = "page_language=?"; $p[] = $flt['lang']; }
        if ($flt['after'])  { $w[] = "last_seen>=?"; $p[] = $flt['after'].'-01'; }
        if ($flt['before']) { $w[] = "last_seen<=?"; $p[] = $flt['before'].'-31'; }
        if (!$w) $w[] = "1=1";
        $ws = implode(' AND ', $w);
        $st = $db->prepare("SELECT COUNT(*) FROM onions WHERE $ws"); $st->execute($p);
        $tot = (int)$st->fetchColumn(); $pgs = max(1, ceil($tot / $LIM));
        $rel = $bt !== '' ? "MATCH(title,meta_description,meta_keywords,tech_stack,tags,notes,cti_name,cti_type,server_software) AGAINST(? IN BOOLEAN MODE)" : "0";
        $sp = $bt !== '' ? array_merge([$bt], $p, [$LIM, $OFF]) : array_merge($p, [$LIM, $OFF]);
        $st = $db->prepare("SELECT address,title,status,cti_type,cti_name,meta_description,tech_stack,server_software,page_language,tags,last_seen,first_seen,page_size,response_time_ms,thumbnail_path,$rel AS rel FROM onions WHERE $ws ORDER BY CASE status WHEN 'online' THEN 0 ELSE 1 END,rel DESC,last_seen DESC LIMIT ? OFFSET ?");
        $st->execute($sp); $res = $st->fetchAll();
    } catch (\Throwable $e) { $err = 'Search error.'; }
    } // end hasAnyFilter

// ════════ NEWEST ════════
} elseif ($V === 'newest') {
    $PT = "Newest Pages — $SN";
    try {
        $tot = (int)$db->query("SELECT COUNT(*) FROM pages")->fetchColumn(); $pgs = max(1, ceil($tot / $LIM));
        $st = $db->prepare("SELECT p.page_url,p.title AS ptitle,p.source,p.first_seen,p.last_seen,o.address,o.title AS stitle,o.status,o.cti_type,o.tags FROM pages p JOIN onions o ON o.id=p.onion_id ORDER BY p.first_seen DESC LIMIT ? OFFSET ?");
        $st->execute([$LIM, $OFF]); $res = $st->fetchAll();
    } catch (\Throwable $e) { $err = 'Error.'; }

// ════════ TAG ════════
} elseif ($V === 'tag' && $TAG !== '') {
    $PT = "Tag: ".htmlspecialchars($TAG)." — $SN";
    $st_tag = '%"'.str_replace(['%','_','"'],['\\%','\\_',''],$TAG).'"%';
    try {
        $st = $db->prepare("SELECT COUNT(*) FROM onions WHERE tags LIKE ?"); $st->execute([$st_tag]);
        $tot = (int)$st->fetchColumn(); $pgs = max(1, ceil($tot / $LIM));
        $st = $db->prepare("SELECT address,title,status,cti_type,cti_name,meta_description,tech_stack,server_software,page_language,tags,last_seen,first_seen,page_size,response_time_ms,thumbnail_path FROM onions WHERE tags LIKE ? ORDER BY CASE status WHEN 'online' THEN 0 ELSE 1 END,last_seen DESC LIMIT ? OFFSET ?");
        $st->execute([$st_tag, $LIM, $OFF]); $res = $st->fetchAll();
    } catch (\Throwable $e) { $err = 'Error.'; }

// ════════ DETAIL ════════
} elseif ($V === 'detail' && $ADDR !== '') {
    try {
        $st = $db->prepare("SELECT * FROM onions WHERE address=?"); $st->execute([$ADDR]);
        $det = $st->fetch();
        if ($det) {
            $PT = htmlspecialchars($det['title'] ?: $ADDR) . " — $SN";
            $st = $db->prepare("SELECT page_url,title,source,first_seen,last_seen FROM pages WHERE onion_id=? ORDER BY first_seen DESC LIMIT 100");
            $st->execute([$det['id']]); $det['_pages'] = $st->fetchAll();
            $st = $db->prepare("SELECT alive,response_ms,checked_at FROM uptime_log WHERE address=? AND checked_at>=DATE_SUB(NOW(),INTERVAL 30 DAY) ORDER BY checked_at ASC LIMIT 90");
            $st->execute([$ADDR]); $det['_uptime'] = $st->fetchAll();
            // Related sites (shared tech + tags)
            $det['_related'] = [];
            $techParts = array_filter(explode(',', $det['tech_stack'] ?? ''));
            $detTags = json_decode($det['tags'] ?? '[]', true) ?: [];
            if ($techParts || $detTags) {
                $rw = []; $rp = [];
                foreach (array_slice($techParts, 0, 3) as $tp) {
                    $rw[] = "tech_stack LIKE ?"; $rp[] = '%'.trim($tp).'%';
                }
                foreach (array_slice($detTags, 0, 3) as $tg) {
                    $rw[] = "tags LIKE ?"; $rp[] = '%"'.$tg.'"%';
                }
                if ($rw) {
                    $rp[] = $ADDR;
                    $st = $db->prepare("SELECT address,title,status,cti_type,tech_stack,tags FROM onions WHERE (".implode(' OR ',$rw).") AND address!=? ORDER BY status='online' DESC, last_seen DESC LIMIT 8");
                    $st->execute($rp); $det['_related'] = $st->fetchAll();
                }
            }
            // Mirrors
            $det['_mirrors'] = [];
            $ch = $det['content_hash'] ?? '';
            if ($ch && strlen($ch) > 8) {
                $st = $db->prepare("SELECT address,title,status FROM onions WHERE content_hash=? AND address!=? LIMIT 10");
                $st->execute([$ch, $ADDR]); $det['_mirrors'] = $st->fetchAll();
            }
            // Snapshots (wayback)
            $st = $db->prepare("SELECT title,content_hash,page_size,captured_at FROM snapshots WHERE address=? ORDER BY captured_at DESC LIMIT 50");
            $st->execute([$ADDR]); $det['_snapshots'] = $st->fetchAll();
            // Canary
            $st = $db->prepare("SELECT * FROM canaries WHERE address=? ORDER BY last_seen DESC LIMIT 1");
            $st->execute([$ADDR]); $det['_canary'] = $st->fetch() ?: null;
        }
    } catch (\Throwable $e) { $err = 'Error.'; }

// ════════ TECH BROWSE ════════
} elseif ($V === 'tech') {
    $TECH = trim($_R['tech'] ?? '');
    $PT = ($TECH ? "Tech: ".htmlspecialchars($TECH) : "Tech Fingerprints") . " — $SN";
    if ($TECH) {
        // Parse "nginx+PHP+WordPress" → AND conditions
        $parts = array_filter(array_map('trim', preg_split('/[+,\s]+/', $TECH)));
        $tw = []; $tp = [];
        foreach ($parts as $p) { $tw[] = "(tech_stack LIKE ? OR server_software LIKE ?)"; $tp[] = "%$p%"; $tp[] = "%$p%"; }
        if ($tw) {
            $ws = implode(' AND ', $tw);
            $st = $db->prepare("SELECT COUNT(*) FROM onions WHERE $ws"); $st->execute($tp);
            $tot = (int)$st->fetchColumn(); $pgs = max(1, ceil($tot / $LIM));
            $st = $db->prepare("SELECT address,title,status,cti_type,tech_stack,server_software,tags,last_seen,meta_description,page_language,response_time_ms,page_size,thumbnail_path,cti_name FROM onions WHERE $ws ORDER BY status='online' DESC,last_seen DESC LIMIT $LIM OFFSET $OFF");
            $st->execute($tp); $res = $st->fetchAll();
        }
    }
    // Top tech breakdown
    try {
        $techTop = $db->query("SELECT server_software, COUNT(*) c FROM onions WHERE server_software!='' GROUP BY server_software ORDER BY c DESC LIMIT 20")->fetchAll();
    } catch (\Throwable $e) { $techTop = []; }

// ════════ LANGUAGE BROWSE ════════
} elseif ($V === 'lang') {
    $LANG = trim($_R['lang'] ?? '');
    $PT = ($LANG ? "Language: ".htmlspecialchars($LANG) : "Languages") . " — $SN";
    if ($LANG) {
        $st = $db->prepare("SELECT COUNT(*) FROM onions WHERE page_language=?"); $st->execute([$LANG]);
        $tot = (int)$st->fetchColumn(); $pgs = max(1, ceil($tot / $LIM));
        $st = $db->prepare("SELECT address,title,status,cti_type,tech_stack,server_software,tags,last_seen,meta_description,page_language,response_time_ms,page_size,thumbnail_path,cti_name FROM onions WHERE page_language=? ORDER BY status='online' DESC,last_seen DESC LIMIT ? OFFSET ?");
        $st->execute([$LANG, $LIM, $OFF]); $res = $st->fetchAll();
    }
    try {
        $langTop = $db->query("SELECT page_language, COUNT(*) c FROM onions WHERE page_language!='' GROUP BY page_language ORDER BY c DESC LIMIT 30")->fetchAll();
    } catch (\Throwable $e) { $langTop = []; }

// ════════ CHANGES FEED ════════
} elseif ($V === 'changes') {
    $PT = "Changes — $SN";
    $cType = $_R['ctype'] ?? '';
    $cDays = min(30, max(1, intval($_R['cdays'] ?? 1)));
    try {
        $cw = ["c.detected_at >= DATE_SUB(NOW(), INTERVAL ? DAY)"]; $cp = [$cDays];
        $validTypes = ['title_changed','came_online','went_offline','new_pages','content_changed','canary_changed'];
        if ($cType && in_array($cType, $validTypes)) { $cw[] = "c.change_type=?"; $cp[] = $cType; }
        $ws = implode(' AND ',$cw);
        // Fix #13: Get actual total count
        $stCnt = $db->prepare("SELECT COUNT(*) FROM site_changes c WHERE $ws");
        $stCnt->execute($cp);
        $tot = (int)$stCnt->fetchColumn();
        $pgs = max(1, ceil($tot / $LIM));
        $cp2 = array_merge($cp, [$LIM, $OFF]);
        $st = $db->prepare("SELECT c.*,o.title AS site_title,o.status,o.cti_type FROM site_changes c LEFT JOIN onions o ON o.address=c.address WHERE $ws ORDER BY c.detected_at DESC LIMIT ? OFFSET ?");
        $st->execute($cp2); $res = $st->fetchAll();
    } catch (\Throwable $e) { $err = 'Error.'; }

// ════════ CANARIES ════════
} elseif ($V === 'canaries') {
    $PT = "Canary Tracker — $SN";
    try {
        $tot = (int)$db->query("SELECT COUNT(*) FROM canaries")->fetchColumn();
        $pgs = max(1, ceil($tot / $LIM));
        $st = $db->prepare("SELECT c.*,o.title AS site_title FROM canaries c LEFT JOIN onions o ON o.address=c.address ORDER BY CASE c.status WHEN 'changed' THEN 0 WHEN 'missing' THEN 1 ELSE 2 END, c.last_seen DESC LIMIT ? OFFSET ?");
        $st->execute([$LIM, $OFF]); $res = $st->fetchAll();
    } catch (\Throwable $e) { $err = 'Error.'; }

// ════════ SUBMIT ════════
} elseif ($V === 'submit') {
    $PT = "Submit a Site — $SN";

// ════════ MIRRORS ════════
} elseif ($V === 'mirrors') {
    $PT = "Mirror Groups — $SN";
    try {
        $tot = (int)$db->query("SELECT COUNT(DISTINCT content_hash) FROM onions WHERE content_hash!='' AND content_hash IN (SELECT content_hash FROM onions WHERE content_hash!='' GROUP BY content_hash HAVING COUNT(*)>1)")->fetchColumn();
        $pgs = max(1, ceil($tot / $LIM));
        $st = $db->prepare("SELECT content_hash, COUNT(*) AS cnt, GROUP_CONCAT(address SEPARATOR '|') AS addrs, GROUP_CONCAT(title SEPARATOR '|') AS titles FROM onions WHERE content_hash!='' GROUP BY content_hash HAVING cnt > 1 ORDER BY cnt DESC LIMIT ? OFFSET ?");
        $st->execute([$LIM, $OFF]);
        $res = $st->fetchAll();
    } catch (\Throwable $e) { $err = 'Error.'; }

} elseif ($V === 'privacy') {
    $PT = "Privacy Policy — $SN";

} elseif ($V === 'terms') {
    $PT = "Terms of Use — $SN";

} elseif ($V === 'removal') {
    $PT = "Removal Request — $SN";
}

// Tag cloud — Fix #16: Cache for 5 minutes instead of rebuilding on every load
if ($V === '' || $V === 'tag') {
    $tagCacheFile = sys_get_temp_dir() . '/inumbra_tagcloud.json';
    $tagCached = false;
    if (file_exists($tagCacheFile) && (time() - filemtime($tagCacheFile)) < 300) {
        $tagCached = json_decode(file_get_contents($tagCacheFile), true);
    }
    if ($tagCached) {
        $allTags = $tagCached;
    } else {
        try {
            $tr = $db->query("SELECT tags FROM onions WHERE tags IS NOT NULL AND tags!='[]' AND tags!='null' LIMIT 2000")->fetchAll(PDO::FETCH_COLUMN);
            $tc = [];
            foreach ($tr as $j) { $a = json_decode($j, true); if (is_array($a)) foreach ($a as $t) { $t=trim($t); if ($t!=='') $tc[$t]=($tc[$t]??0)+1; } }
            arsort($tc); $allTags = array_slice($tc, 0, 60, true);
            @file_put_contents($tagCacheFile, json_encode($allTags));
        } catch (\Throwable $e) {}
    }
}
// Trending
if ($V === '') {
    try { $trend = $db->query("SELECT term,hits FROM trending_terms WHERE approved=1 ORDER BY hits DESC LIMIT 10")->fetchAll(); } catch (\Throwable $e) {}
}
// Stats — Fix #15: Cache counts for 60 seconds to avoid full table scans
$stats = ['total'=>0,'online'=>0,'pages'=>0];
if ($V === '') {
    $cacheFile = sys_get_temp_dir() . '/inumbra_stats.json';
    $cached = false;
    if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < 60) {
        $cached = json_decode(file_get_contents($cacheFile), true);
    }
    if ($cached) {
        $stats = $cached;
    } else {
        try {
            $stats['total']=(int)$db->query("SELECT COUNT(*) FROM onions")->fetchColumn();
            $stats['online']=(int)$db->query("SELECT COUNT(*) FROM onions WHERE status='online'")->fetchColumn();
            $stats['pages']=(int)$db->query("SELECT COUNT(*) FROM pages")->fetchColumn();
            @file_put_contents($cacheFile, json_encode($stats));
        } catch (\Throwable $e) {}
    }
}
// Sparklines for result lists
if (($V === 'search' || $V === 'tag') && $res) {
    $addrs = array_column($res, 'address');
    if ($addrs) {
        $ph = implode(',', array_fill(0, count($addrs), '?'));
        try {
            $st = $db->prepare("SELECT address,alive,checked_at FROM uptime_log WHERE address IN ($ph) AND checked_at>=DATE_SUB(NOW(),INTERVAL 30 DAY) ORDER BY checked_at ASC");
            $st->execute($addrs);
            foreach ($st->fetchAll() as $row) $sparks[$row['address']][] = (int)$row['alive'];
        } catch (\Throwable $e) {}
    }
}
$hQ = htmlspecialchars($Q); $hT = htmlspecialchars($TAG); $hA = htmlspecialchars($ADDR);
?>
<?php
// ── Dynamic SEO meta ──
$siteUrl = rtrim($cfg['site_url'] ?? 'https://inumbra.com', '/');
$metaDesc = "In Umbra — dark web meta-search engine indexing " . number_format($stats['total'] ?? 0) . " .onion hidden services. Search by status, technology stack, language, CTI classification, and more. Free, open OSINT tool for security researchers.";
$canonical = $siteUrl;
$httpStatus = 200;
if ($V === 'search' && $Q !== '') {
    $metaDesc = "Dark web search results for \"" . htmlspecialchars(mb_substr($Q, 0, 80)) . "\" — In Umbra indexes .onion hidden services with advanced filters for OSINT research.";
    $PT = htmlspecialchars(mb_substr($Q, 0, 60)) . " — In Umbra Search";
    $canonical = $siteUrl . '/?v=search&q=' . urlencode($Q);
} elseif ($V === 'detail' && $det) {
    $metaDesc = "Detailed OSINT profile for " . htmlspecialchars(mb_substr($det['title'] ?: $ADDR, 0, 80)) . " — status, technology stack, uptime history, sub-pages, mirrors, and wayback snapshots.";
    $PT = htmlspecialchars(mb_substr($det['title'] ?: $ADDR, 0, 60)) . " — In Umbra";
    $canonical = $siteUrl . '/?v=detail&addr=' . urlencode($ADDR);
} elseif ($V === 'detail' && !$det) {
    $httpStatus = 404;
} elseif ($V === 'newest') {
    $metaDesc = "Recently discovered dark web pages and hidden services. Updated continuously by In Umbra's crawler.";
    $PT = "Newest Pages — In Umbra";
    $canonical = $siteUrl . '/?v=newest';
} elseif ($V === 'changes') {
    $metaDesc = "Real-time change detection feed for dark web hidden services. Track when .onion sites come online, go offline, or change titles.";
    $PT = "Change Feed — In Umbra";
    $canonical = $siteUrl . '/?v=changes';
} elseif ($V === 'tech') {
    $metaDesc = "Browse dark web sites by server technology stack. Search by nginx, Apache, PHP, WordPress, and more. Fingerprint .onion infrastructure.";
    $PT = "Technology Browser — In Umbra";
    $canonical = $siteUrl . '/?v=tech';
} elseif ($V === 'lang') {
    $metaDesc = "Browse dark web sites by detected language. Filter .onion hidden services across 30+ languages.";
    $PT = "Language Browser — In Umbra";
    $canonical = $siteUrl . '/?v=lang';
} elseif ($V === 'canaries') {
    $metaDesc = "Warrant canary tracker for dark web hidden services. Monitor PGP-signed canary statements and detect when they change or disappear.";
    $PT = "Canary Tracker — In Umbra";
    $canonical = $siteUrl . '/?v=canaries';
} elseif ($V === 'mirrors') {
    $metaDesc = "Detect mirror sites on the dark web. Groups .onion hidden services with identical content using cryptographic hash comparison.";
    $PT = "Mirror Detection — In Umbra";
    $canonical = $siteUrl . '/?v=mirrors';
} elseif ($V === 'submit') {
    $metaDesc = "Submit a .onion hidden service to be indexed by In Umbra. All submissions are reviewed before crawling.";
    $PT = "Submit a Site — In Umbra";
    $canonical = $siteUrl . '/?v=submit';
} elseif ($V === 'tag' && $TAG) {
    $metaDesc = "Dark web sites tagged \"" . htmlspecialchars($TAG) . "\" — browse classified .onion hidden services on In Umbra.";
    $PT = htmlspecialchars($TAG) . " — In Umbra";
    $canonical = $siteUrl . '/?v=tag&tag=' . urlencode($TAG);
} elseif ($V === 'privacy') {
    $metaDesc = "In Umbra privacy policy. How we collect, use, and protect data. Details on analytics, search logs, IP hashing, and data retention for our dark web meta-search engine.";
    $PT = "Privacy Policy — In Umbra";
    $canonical = $siteUrl . '/?v=privacy';
} elseif ($V === 'terms') {
    $metaDesc = "In Umbra terms of use. Legal terms governing use of our dark web meta-search engine, content indexing, DMCA and takedown procedures, and limitation of liability.";
    $PT = "Terms of Use — In Umbra";
    $canonical = $siteUrl . '/?v=terms';
} elseif ($V === 'removal') {
    $metaDesc = "Request removal of a .onion address or personal information from In Umbra's dark web index. Site operators and individuals can submit delisting requests.";
    $PT = "Removal Request — In Umbra";
    $canonical = $siteUrl . '/?v=removal';
} elseif ($V === '') {
    $PT = "In Umbra — Dark Web Meta-Search Engine for OSINT Research";
} else {
    // Fix #14: Invalid view parameter — redirect to home
    $V = '';
    $PT = "In Umbra — Dark Web Meta-Search Engine for OSINT Research";
    $httpStatus = 404;
}
// Fix #26: Set HTTP status for 404 pages
if ($httpStatus !== 200) { http_response_code($httpStatus); }
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
<meta name="description" content="<?= htmlspecialchars($metaDesc) ?>">
<meta name="robots" content="index,follow">
<link rel="canonical" href="<?= htmlspecialchars($canonical) ?>">

<!-- Open Graph -->
<meta property="og:type" content="website">
<meta property="og:site_name" content="In Umbra">
<meta property="og:title" content="<?= htmlspecialchars($PT) ?>">
<meta property="og:description" content="<?= htmlspecialchars($metaDesc) ?>">
<meta property="og:url" content="<?= htmlspecialchars($canonical) ?>">
<meta property="og:image" content="<?= $siteUrl ?>/assets/logo.svg">

<!-- Twitter Card -->
<meta name="twitter:card" content="summary">
<meta name="twitter:title" content="<?= htmlspecialchars($PT) ?>">
<meta name="twitter:description" content="<?= htmlspecialchars($metaDesc) ?>">

<!-- JSON-LD Schema -->
<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@graph": [
    {
      "@type": "Organization",
      "name": "In Umbra",
      "url": "<?= $siteUrl ?>",
      "logo": "<?= $siteUrl ?>/assets/logo.svg",
      "description": "Dark web meta-search engine and OSINT research tool. Indexes .onion hidden services with advanced filtering, change detection, mirror detection, warrant canary tracking, and technology fingerprinting."
    },
    {
      "@type": "WebSite",
      "name": "In Umbra",
      "url": "<?= $siteUrl ?>",
      "description": "Search engine for Tor hidden services (.onion sites). Provides metadata indexing, uptime monitoring, technology fingerprinting, and threat intelligence classification for OSINT researchers.",
      "potentialAction": {
        "@type": "SearchAction",
        "target": "<?= $siteUrl ?>/?v=search&q={search_term_string}",
        "query-input": "required name=search_term_string"
      }
    },
    {
      "@type": "FAQPage",
      "mainEntity": [
        {
          "@type": "Question",
          "name": "What is In Umbra?",
          "acceptedAnswer": {
            "@type": "Answer",
            "text": "In Umbra is a dark web meta-search engine that indexes .onion hidden services on the Tor network. It provides metadata-only indexing — titles, server technology, uptime status, language detection, and threat intelligence classifications — without storing or caching any actual page content. It is designed as an OSINT (Open Source Intelligence) tool for security researchers, journalists, and threat analysts."
          }
        },
        {
          "@type": "Question",
          "name": "How does In Umbra detect dark web sites?",
          "acceptedAnswer": {
            "@type": "Answer",
            "text": "In Umbra uses a multi-source discovery pipeline: it imports known .onion addresses from public directories (Ahmia, Fresh Onions, Tor66), performs BFS (breadth-first search) crawling to discover linked sites, and accepts user submissions. Each discovered site is periodically checked for status, and metadata is extracted including server software, technology stack, page language, and content hashes for mirror detection."
          }
        },
        {
          "@type": "Question",
          "name": "What is warrant canary tracking?",
          "acceptedAnswer": {
            "@type": "Answer",
            "text": "A warrant canary is a public statement, often PGP-signed, that a service provider has not received secret government subpoenas or gag orders. In Umbra's canary tracker automatically detects these statements on .onion sites, hashes them, and monitors for changes. If a canary is modified or disappears, it may indicate the site has been served with a legal order — a critical signal for security researchers."
          }
        },
        {
          "@type": "Question",
          "name": "How does mirror detection work on the dark web?",
          "acceptedAnswer": {
            "@type": "Answer",
            "text": "In Umbra computes a normalized SHA-256 hash of each site's content after stripping dynamic elements like scripts, nonces, timestamps, and session tokens. Sites with identical content hashes are grouped as mirrors. This helps researchers identify redundant infrastructure, track site operators across multiple addresses, and detect when a site moves to a new .onion address."
          }
        },
        {
          "@type": "Question",
          "name": "What advanced search filters does In Umbra support?",
          "acceptedAnswer": {
            "@type": "Answer",
            "text": "In Umbra supports the following search syntax: status:online or status:offline to filter by availability, cti:ransomware or cti:marketplace for threat intelligence classification, tech:nginx or tech:apache for server technology, lang:ru or lang:de for page language, and after:2025-01 or before:2024-06 for date filtering. These can be combined with free-text keywords."
          }
        },
        {
          "@type": "Question",
          "name": "Does In Umbra store dark web content?",
          "acceptedAnswer": {
            "@type": "Answer",
            "text": "No. In Umbra is a metadata-only index. It stores titles, server headers, technology fingerprints, uptime history, content hashes, and structural metadata. It does not store, cache, or proxy any actual page content from .onion sites. The search engine applies the Ahmia CSAM blocklist to filter known child abuse material and blocks reported illegal content."
          }
        }
      ]
    }<?php if ($V === 'detail' && $det): ?>,
    {
      "@type": "Article",
      "headline": "<?= htmlspecialchars($det['title'] ?: $ADDR) ?> — Dark Web Site Profile",
      "dateModified": "<?= htmlspecialchars($det['last_seen'] ?? '') ?>",
      "description": "<?= htmlspecialchars(mb_substr($det['meta_description'] ?: 'OSINT profile for ' . $ADDR, 0, 200)) ?>",
      "publisher": { "@type": "Organization", "name": "In Umbra" }
    }<?php endif; ?>
  ]
}
</script>

<link rel="alternate" type="application/atom+xml" title="Newest Sites" href="api/feed.php?type=sites">
<link rel="alternate" type="application/atom+xml" title="Newest Pages" href="api/feed.php?type=pages">
<link rel="icon" type="image/svg+xml" href="assets/icon.svg">
<title><?= $PT ?></title>
<style>
*{margin:0;padding:0;box-sizing:border-box}
:root{--bg:#0a0a0f;--s:#12121a;--s2:#1a1a25;--bd:#252535;--t:#c8c8d8;--dm:#555570;
--ac:#7b68ee;--ac2:#9b88ff;--on:#22c55e;--off:#666;--tor:#b388ff;--w:#ef4444;
--cm:#e67e22;--cf:#3498db;--cr:#e74c3c}
body{background:var(--bg);color:var(--t);font-family:'Courier New',Consolas,monospace;min-height:100vh}
a{color:var(--ac);text-decoration:none}a:hover{text-decoration:underline}
.home{display:flex;flex-direction:column;align-items:center;justify-content:center;min-height:78vh}
.home h1{font-size:3.2rem;letter-spacing:.15em;color:var(--ac);margin-bottom:.2em;font-weight:300}
.home .tl{color:var(--dm);font-size:.8rem;margin-bottom:2em;letter-spacing:.3em;text-transform:uppercase}
.home .st{color:var(--dm);font-size:.75rem;margin-top:1.5em;letter-spacing:.1em}.home .st b{color:var(--ac)}
.nb{background:var(--s);border:1px solid var(--bd);color:var(--dm);padding:8px 18px;font-family:inherit;font-size:.75rem;cursor:pointer;border-radius:4px;transition:all .15s}
.nb:hover{border-color:var(--ac);color:var(--ac)}.nb.on{background:var(--ac);color:var(--bg);border-color:var(--ac)}
.sf{display:flex;gap:0;max-width:620px;width:100%}
.sf input[type=text]{flex:1;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:14px 18px;font-size:1rem;font-family:inherit;outline:none;border-radius:6px 0 0 6px;transition:border-color .2s}
.sf input:focus{border-color:var(--ac)}
.sf button[type=submit]{background:var(--ac);color:var(--bg);border:none;padding:14px 28px;font-size:1rem;font-family:inherit;font-weight:bold;cursor:pointer;border-radius:0 6px 6px 0;transition:background .2s}
.sf button[type=submit]:hover{background:var(--ac2)}
.rp{max-width:860px;margin:0 auto;padding:20px}
.tb{display:flex;align-items:center;gap:12px;padding:10px 0 16px;border-bottom:1px solid var(--bd);margin-bottom:16px;flex-wrap:wrap}
.lb{background:none;border:none;color:var(--ac);font:inherit;font-size:1.1rem;font-weight:300;letter-spacing:.1em;cursor:pointer;padding:0}
.tb .sf{max-width:380px;flex:1}.tb .sf input{padding:10px 14px;font-size:.85rem}.tb .sf button[type=submit]{padding:10px 18px;font-size:.85rem}
.tn{display:flex;gap:6px}.tn .nb{padding:6px 14px;font-size:.7rem}
.mi{color:var(--dm);font-size:.75rem;margin-bottom:16px}
.card{margin-bottom:18px;padding-bottom:14px;border-bottom:1px solid var(--bd)}.card:last-child{border-bottom:none}
.ch{display:flex;align-items:center;gap:8px;margin-bottom:3px;flex-wrap:wrap}
.badge{display:inline-block;font-size:.65rem;padding:2px 7px;border-radius:3px;font-weight:bold;text-transform:uppercase}
.b-on{background:var(--on);color:#000}.b-off{background:var(--s2);color:var(--dm)}.b-cti{background:var(--cm);color:#fff}.b-ran{background:var(--cr);color:#fff}.b-for{background:var(--cf);color:#fff}
.ct{font-size:1rem;font-weight:bold;color:var(--t);margin-bottom:2px;cursor:pointer;background:none;border:none;font-family:inherit;text-align:left;padding:0}.ct:hover{color:var(--ac)}
.ca{font-size:.8rem;color:var(--tor);word-break:break-all}
.cd{font-size:.8rem;color:var(--dm);margin-top:4px;line-height:1.5}
.cm{display:flex;gap:12px;margin-top:6px;font-size:.7rem;color:var(--dm);flex-wrap:wrap}.cm span{white-space:nowrap}
.tl2{display:flex;gap:4px;flex-wrap:wrap;margin-top:5px}
.tp{font-size:.6rem;padding:2px 8px;border-radius:10px;background:var(--s2);color:var(--ac);border:1px solid var(--bd);cursor:pointer;transition:all .15s;font-family:inherit}
.tp:hover{border-color:var(--ac);background:var(--ac);color:var(--bg)}.tp.on{background:var(--ac);color:var(--bg);border-color:var(--ac)}
.tc{display:flex;flex-wrap:wrap;gap:6px;justify-content:center;margin-top:1.8em;max-width:620px}.tc .tp{font-size:.7rem;padding:4px 12px}
.spark{display:inline-flex;align-items:end;gap:1px;height:14px;vertical-align:middle;margin-left:6px}
.spark i{display:block;width:3px;border-radius:1px;min-height:2px}
.spark-lg{display:flex;align-items:end;gap:2px;height:30px;margin:8px 0}.spark-lg i{display:block;width:6px;border-radius:2px;min-height:3px}
.pc{margin-bottom:12px;padding:12px 14px;background:var(--s);border:1px solid var(--bd);border-radius:6px}
.pager{display:flex;justify-content:center;gap:4px;margin-top:24px}
.pager button{font-family:inherit;font-size:.8rem;padding:6px 14px;border:1px solid var(--bd);background:var(--s);color:var(--dm);cursor:pointer;border-radius:4px;transition:all .15s}
.pager button:hover{border-color:var(--ac);color:var(--ac)}.pager button.on{background:var(--ac);color:var(--bg);border-color:var(--ac);font-weight:bold}
.rb{background:none;border:none;color:var(--dm);font-family:inherit;font-size:.65rem;cursor:pointer;padding:2px 0;opacity:.6;transition:all .15s}.rb:hover{color:var(--w);opacity:1}
.rx{display:none;margin-top:8px;padding:10px;background:var(--s2);border:1px solid var(--bd);border-radius:6px}.rx.open{display:block}
.rx select,.rx textarea{font-family:inherit;font-size:.75rem;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:6px 8px;border-radius:4px;width:100%;margin-bottom:6px}.rx textarea{resize:vertical;min-height:40px}
.rs{background:var(--w);color:#fff;border:none;font-family:inherit;font-size:.7rem;padding:6px 16px;border-radius:4px;cursor:pointer;font-weight:bold}
.rok{color:var(--on);font-size:.75rem;font-weight:bold}
.dt{background:var(--s);border:1px solid var(--bd);border-radius:8px;padding:16px;margin-bottom:12px}
.dt h2{font-size:1.1rem;color:var(--t);margin-bottom:4px}.dt .da{color:var(--tor);font-size:.85rem;word-break:break-all;margin-bottom:10px}
.dg{display:grid;grid-template-columns:1fr 1fr;gap:6px 20px;font-size:.8rem}.dg .k{color:var(--dm)}.dg .v{color:var(--t);word-break:break-all}
.ds{margin-top:16px;padding-top:12px;border-top:1px solid var(--bd)}.ds h3{font-size:.9rem;color:var(--ac);margin-bottom:8px}
.up-st{font-size:.75rem;color:var(--dm);margin-top:4px}
.trend{display:flex;flex-wrap:wrap;gap:6px;justify-content:center;margin-top:1em;max-width:620px}.trend .nb{font-size:.7rem;padding:4px 14px}
.footer{text-align:center;color:var(--dm);font-size:.65rem;padding:30px;letter-spacing:.1em}.footer a{color:var(--dm)}.footer a:hover{color:var(--ac)}
.empty{color:var(--dm);padding:40px 0;text-align:center}
.adv{font-size:.65rem;color:var(--dm);margin-top:.6em;text-align:center}.adv code{background:var(--s2);padding:1px 5px;border-radius:3px;color:var(--ac)}
.subp{font-size:.8rem;color:var(--tor);word-break:break-all;padding:4px 0;border-bottom:1px solid var(--bd)}
.subp:last-child{border-bottom:none}
.subp-t{color:var(--t);font-size:.75rem}
</style>
<script>
var _csrf='<?= htmlspecialchars($csrfToken, ENT_QUOTES) ?>';
function tr(id){document.getElementById('rpt-'+id).classList.toggle('open')}
function sr(id,a){var r=document.getElementById('rpt-'+id),b=r.querySelector('.rs');b.textContent='…';b.disabled=true;
fetch('api/report.php',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({address:a,reason:r.querySelector('select').value,detail:r.querySelector('textarea').value,_csrf:_csrf})})
.then(function(x){return x.json()}).then(function(d){r.innerHTML='<span class="rok">✓ '+(d.message||d.status)+'</span>';}).catch(function(){b.textContent='Report';b.disabled=false;});}
// Auto-add CSRF hidden input to all forms on page
document.addEventListener('DOMContentLoaded',function(){document.querySelectorAll('form').forEach(function(f){if(!f.querySelector('[name=_csrf]')){var i=document.createElement('input');i.type='hidden';i.name='_csrf';i.value=_csrf;f.appendChild(i);}});});
</script>
</head>
<body>

<?php if ($V === ''): ?>
<!-- ═══════ HOME ═══════ -->
<main class="home" role="main">
  <header>
    <h1><?= $SN ?></h1>
    <p class="tl"><?= htmlspecialchars($cfg['site_tagline']) ?></p>
  </header>
  <form class="sf" method="post" role="search" aria-label="Search dark web index"><input type="hidden" name="v" value="search">
    <input type="text" name="q" placeholder="Search the dark web index…" autofocus autocomplete="off" aria-label="Search query">
    <button type="submit">Search</button></form>
  <div class="adv">Try: <code>status:online</code> <code>cti:ransomware</code> <code>tech:nginx</code> <code>lang:ru</code> <code>after:2025-01</code></div>
  <?php if ($stats['total']): ?><div class="st">Indexing <b><?= number_format($stats['total']) ?></b> services · <b><?= number_format($stats['online']) ?></b> online · <b><?= number_format($stats['pages']) ?></b> pages</div><?php endif; ?>
  <nav style="display:flex;gap:6px;margin-top:1.2em;flex-wrap:wrap;justify-content:center" aria-label="Browse categories">
    <form method="post"><input type="hidden" name="v" value="newest"><button type="submit" class="nb">📄 Newest</button></form>
    <form method="post"><input type="hidden" name="v" value="changes"><button type="submit" class="nb">🔄 Changes</button></form>
    <form method="post"><input type="hidden" name="v" value="tech"><button type="submit" class="nb">⚙ Tech</button></form>
    <form method="post"><input type="hidden" name="v" value="lang"><button type="submit" class="nb">🌐 Languages</button></form>
    <form method="post"><input type="hidden" name="v" value="mirrors"><button type="submit" class="nb">🪞 Mirrors</button></form>
    <form method="post"><input type="hidden" name="v" value="canaries"><button type="submit" class="nb">🐤 Canaries</button></form>
    <form method="post"><input type="hidden" name="v" value="submit"><button type="submit" class="nb">➕ Submit</button></form>
  </nav>
  <?php if ($trend): ?>
  <div class="trend"><?php foreach ($trend as $tr): ?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="search"><input type="hidden" name="q" value="<?= htmlspecialchars($tr['term']) ?>">
      <button type="submit" class="nb">🔥 <?= htmlspecialchars($tr['term']) ?></button></form>
  <?php endforeach; ?></div><?php endif; ?>
  <?php if ($allTags): ?>
  <div class="tc"><?php foreach ($allTags as $t => $c): ?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="tag"><input type="hidden" name="tag" value="<?= htmlspecialchars($t) ?>">
      <button type="submit" class="tp"><?= htmlspecialchars($t) ?> <span style="opacity:.5"><?= $c ?></span></button></form>
  <?php endforeach; ?></div><?php endif; ?>

  <!-- About / FAQ Section — visible, structured for AI extraction -->
  <section class="about-section" style="max-width:680px;margin-top:3em;text-align:left;width:100%;padding:0 20px">
    <h2 style="font-size:1rem;color:var(--ac);letter-spacing:.1em;margin-bottom:1em;text-align:center">About In Umbra</h2>
    <p style="font-size:.8rem;color:var(--dm);line-height:1.7;margin-bottom:1.2em">
      In Umbra is a metadata-only dark web search engine that indexes .onion hidden services on the Tor network. Unlike traditional search engines, In Umbra does not store or cache page content. It indexes titles, server technology, uptime status, content hashes, threat intelligence classifications, and page language — providing OSINT researchers with structured data about the dark web without hosting any of its content.
    </p>
    <p style="font-size:.8rem;color:var(--dm);line-height:1.7;margin-bottom:1.5em">
      The index is built from multiple sources: public .onion directories, breadth-first crawling, and user submissions. Every site is checked against the Ahmia CSAM blocklist. Features include real-time change detection, warrant canary monitoring, content-hash-based mirror detection, technology stack fingerprinting, and a wayback-style snapshot timeline.
    </p>

    <h2 style="font-size:1rem;color:var(--ac);letter-spacing:.1em;margin-bottom:1em;text-align:center">Frequently Asked Questions</h2>

    <details style="margin-bottom:10px;border:1px solid var(--bd);border-radius:6px;padding:0">
      <summary style="padding:12px 16px;cursor:pointer;font-size:.85rem;color:var(--t);background:var(--s);border-radius:6px">What is In Umbra?</summary>
      <div style="padding:12px 16px;font-size:.8rem;color:var(--dm);line-height:1.7">In Umbra is a dark web meta-search engine that indexes .onion hidden services on the Tor network. It provides metadata-only indexing — titles, server technology, uptime status, language detection, and threat intelligence classifications — without storing or caching any actual page content. It is designed as an OSINT (Open Source Intelligence) tool for security researchers, journalists, and threat analysts.</div>
    </details>

    <details style="margin-bottom:10px;border:1px solid var(--bd);border-radius:6px;padding:0">
      <summary style="padding:12px 16px;cursor:pointer;font-size:.85rem;color:var(--t);background:var(--s);border-radius:6px">How does In Umbra discover .onion sites?</summary>
      <div style="padding:12px 16px;font-size:.8rem;color:var(--dm);line-height:1.7">In Umbra uses a multi-source discovery pipeline: it imports known .onion addresses from public directories (Ahmia, Fresh Onions, Tor66), performs BFS (breadth-first search) crawling to discover linked sites, and accepts user submissions. Each discovered site is periodically checked for status, and metadata is extracted including server software, technology stack, page language, and content hashes for mirror detection.</div>
    </details>

    <details style="margin-bottom:10px;border:1px solid var(--bd);border-radius:6px;padding:0">
      <summary style="padding:12px 16px;cursor:pointer;font-size:.85rem;color:var(--t);background:var(--s);border-radius:6px">What is warrant canary tracking?</summary>
      <div style="padding:12px 16px;font-size:.8rem;color:var(--dm);line-height:1.7">A warrant canary is a public statement, often PGP-signed, declaring that a service provider has not received secret government subpoenas or gag orders. In Umbra's canary tracker automatically detects these statements on .onion sites, hashes them cryptographically, and monitors for changes. If a canary is modified or disappears, it may indicate the site has been served with a legal order — a critical signal for security researchers and privacy advocates.</div>
    </details>

    <details style="margin-bottom:10px;border:1px solid var(--bd);border-radius:6px;padding:0">
      <summary style="padding:12px 16px;cursor:pointer;font-size:.85rem;color:var(--t);background:var(--s);border-radius:6px">How does mirror detection work?</summary>
      <div style="padding:12px 16px;font-size:.8rem;color:var(--dm);line-height:1.7">In Umbra computes a normalized SHA-256 hash of each site's content after stripping dynamic elements like scripts, nonces, timestamps, and session tokens. Sites with identical content hashes are grouped as mirrors. This helps researchers identify redundant infrastructure, track site operators across multiple .onion addresses, and detect when a site migrates to a new address.</div>
    </details>

    <details style="margin-bottom:10px;border:1px solid var(--bd);border-radius:6px;padding:0">
      <summary style="padding:12px 16px;cursor:pointer;font-size:.85rem;color:var(--t);background:var(--s);border-radius:6px">What search filters are available?</summary>
      <div style="padding:12px 16px;font-size:.8rem;color:var(--dm);line-height:1.7">In Umbra supports advanced search syntax: <code>status:online</code> or <code>status:offline</code> for availability, <code>cti:ransomware</code> or <code>cti:marketplace</code> for threat intelligence, <code>tech:nginx</code> for server technology, <code>lang:ru</code> for language, and <code>after:2025-01</code> or <code>before:2024-06</code> for date ranges. Combine any of these with free-text keywords for precise results.</div>
    </details>

    <details style="margin-bottom:10px;border:1px solid var(--bd);border-radius:6px;padding:0">
      <summary style="padding:12px 16px;cursor:pointer;font-size:.85rem;color:var(--t);background:var(--s);border-radius:6px">Does In Umbra store dark web content?</summary>
      <div style="padding:12px 16px;font-size:.8rem;color:var(--dm);line-height:1.7">No. In Umbra is strictly a metadata-only index. It stores titles, server headers, technology fingerprints, uptime history, content hashes, and structural metadata — never actual page content. The Ahmia CSAM blocklist is enforced on all ingested data, and users can report sites for review. This approach provides research value while avoiding the legal and ethical issues of hosting dark web content.</div>
    </details>
  </section>
</main>

<?php else: ?>
<!-- ═══════ TOP BAR ═══════ -->
<div class="rp">
  <nav class="tb" aria-label="Site navigation">
    <form method="post"><button type="submit" class="lb"><?= $SN ?></button></form>
    <form class="sf" method="post" role="search" aria-label="Search"><input type="hidden" name="v" value="search">
      <input type="text" name="q" value="<?= $V==='search'?$hQ:'' ?>" placeholder="Search…" autocomplete="off" aria-label="Search query">
      <button type="submit">Search</button></form>
    <div class="tn">
      <form method="post" style="display:inline"><input type="hidden" name="v" value="newest"><button type="submit" class="nb <?= $V==='newest'?'on':'' ?>">📄</button></form>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="changes"><button type="submit" class="nb <?= $V==='changes'?'on':'' ?>">🔄</button></form>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="tech"><button type="submit" class="nb <?= $V==='tech'?'on':'' ?>">⚙</button></form>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="lang"><button type="submit" class="nb <?= $V==='lang'?'on':'' ?>">🌐</button></form>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="submit"><button type="submit" class="nb <?= $V==='submit'?'on':'' ?>">➕</button></form>
      <a href="api/feed.php?type=sites" class="nb" style="text-decoration:none">📡</a>
    </div>
  </nav>
  <main role="main">

<?php if ($V === 'search'): ?>
<!-- ═══════ SEARCH RESULTS ═══════ -->
  <?php if ($err): ?><div style="color:var(--w);padding:20px"><?= $err ?></div>
  <?php elseif (!$res): ?><div class="empty">No results for <b>"<?= $hQ ?>"</b></div>
  <?php else: ?>
    <div class="mi"><?= number_format($tot) ?> results · page <?= $PG ?>/<?= $pgs ?></div>
    <?php foreach ($res as $r): ?>
    <div class="card">
      <div class="ch">
        <span class="badge <?= ($r['status']??'')==='online'?'b-on':'b-off' ?>"><?= $r['status'] ?></span>
        <?php $cti=$r['cti_type']??''; if ($cti): $cc=stripos($cti,'ransom')!==false?'b-ran':(stripos($cti,'forum')!==false?'b-for':'b-cti'); ?>
          <span class="badge <?= $cc ?>"><?= htmlspecialchars($cti) ?></span><?php endif; ?>
        <?php if ($r['cti_name']??''): ?><span style="font-size:.7rem;color:var(--w)"><?= htmlspecialchars(mb_substr($r['cti_name'],0,40)) ?></span><?php endif; ?>
        <?php if (!empty($sparks[$r['address']])): ?><?= spark($sparks[$r['address']]) ?><?php endif; ?>
      </div>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?= htmlspecialchars($r['address']) ?>">
        <button type="submit" class="ct"><?= htmlspecialchars($r['title'] ?: '(untitled)') ?></button></form>
      <div class="ca"><?= htmlspecialchars($r['address']) ?></div>
      <?php if ($r['meta_description']??''): ?><div class="cd"><?= htmlspecialchars(mb_substr($r['meta_description'],0,250)) ?></div><?php endif; ?>
      <div class="cm">
        <?php if ($r['server_software']??''): ?><span>🖥 <?= htmlspecialchars($r['server_software']) ?></span><?php endif; ?>
        <?php if ($r['tech_stack']??''): ?><span>⚙ <?= htmlspecialchars(mb_substr($r['tech_stack'],0,60)) ?></span><?php endif; ?>
        <?php if ($r['page_language']??''): ?><span>🌐 <?= htmlspecialchars($r['page_language']) ?></span><?php endif; ?>
        <?php if ($r['page_size']??0): ?><span><?= number_format($r['page_size']) ?>b</span><?php endif; ?>
        <?php if ($r['response_time_ms']??0): ?><span><?= $r['response_time_ms'] ?>ms</span><?php endif; ?>
        <?php if ($r['last_seen']??''): ?><span>seen <?= substr($r['last_seen'],0,10) ?></span><?php endif; ?>
      </div>
      <?php $tags=json_decode($r['tags']?:'[]',true); if ($tags): ?>
      <div class="tl2"><?php foreach (array_slice($tags,0,6) as $t): ?>
        <form method="post" style="display:inline"><input type="hidden" name="v" value="tag"><input type="hidden" name="tag" value="<?= htmlspecialchars($t) ?>"><button type="submit" class="tp"><?= htmlspecialchars($t) ?></button></form>
      <?php endforeach; ?></div><?php endif; ?>
      <?php $_ri++; ?><button class="rb" onclick="tr(<?=$_ri?>)">⚑ report</button>
      <div class="rx" id="rpt-<?=$_ri?>"><select><option value="csam">CSAM</option><option value="scam">Scam</option><option value="malware">Malware</option><option value="miscat">Miscategorized</option><option value="other">Other</option></select><textarea placeholder="Details…" maxlength="500"></textarea><button class="rs" onclick="sr(<?=$_ri?>,'<?=htmlspecialchars($r['address'], ENT_QUOTES)?>')">Submit</button></div>
    </div>
    <?php endforeach; ?>
    <?php if ($pgs > 1): ?><div class="pager"><?php for ($i=max(1,$PG-3);$i<=min($pgs,$PG+3);$i++): ?>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="search"><input type="hidden" name="q" value="<?=$hQ?>"><input type="hidden" name="p" value="<?=$i?>"><button type="submit" class="<?=$i===$PG?'on':''?>"><?=$i?></button></form>
    <?php endfor; ?></div><?php endif; ?>
  <?php endif; ?>

<?php elseif ($V === 'newest'): ?>
<!-- ═══════ NEWEST ═══════ -->
  <div class="mi">📄 <?= number_format($tot) ?> pages discovered · page <?= $PG ?>/<?= $pgs ?></div>
  <?php if ($err): ?><div style="color:var(--w)"><?= $err ?></div>
  <?php elseif (!$res): ?><div class="empty">No pages yet.</div>
  <?php else: foreach ($res as $r): ?>
    <div class="pc">
      <div class="ch">
        <span class="badge <?=($r['status']??'')==='online'?'b-on':'b-off'?>"><?=$r['status']?></span>
        <?php if($r['cti_type']??''):?><span class="badge b-cti"><?=htmlspecialchars($r['cti_type'])?></span><?php endif;?>
        <span style="font-size:.65rem;color:var(--dm)">found <?=substr($r['first_seen'],0,16)?></span>
      </div>
      <?php if ($r['ptitle']??''): ?><div style="font-size:.9rem;font-weight:bold;color:var(--t);margin-bottom:2px"><?= htmlspecialchars(mb_substr($r['ptitle'],0,120)) ?></div><?php endif; ?>
      <div style="font-size:.8rem;color:var(--tor);word-break:break-all"><?= htmlspecialchars(mb_substr($r['page_url'],0,150)) ?></div>
      <div style="font-size:.75rem;color:var(--dm)">on <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($r['address'])?>"><button type="submit" style="background:none;border:none;color:var(--ac);font:inherit;cursor:pointer;padding:0"><b><?=htmlspecialchars($r['stitle']?:$r['address'])?></b></button></form></div>
      <?php $ptags=json_decode($r['tags']?:'[]',true); if($ptags):?>
      <div class="tl2" style="margin-top:4px"><?php foreach(array_slice($ptags,0,5) as $t):?>
        <form method="post" style="display:inline"><input type="hidden" name="v" value="tag"><input type="hidden" name="tag" value="<?=htmlspecialchars($t)?>"><button type="submit" class="tp"><?=htmlspecialchars($t)?></button></form>
      <?php endforeach;?></div><?php endif;?>
    </div>
  <?php endforeach;
    if ($pgs > 1): ?><div class="pager"><?php for ($i=max(1,$PG-3);$i<=min($pgs,$PG+3);$i++): ?>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="newest"><input type="hidden" name="p" value="<?=$i?>"><button type="submit" class="<?=$i===$PG?'on':''?>"><?=$i?></button></form>
    <?php endfor; ?></div><?php endif; endif; ?>

<?php elseif ($V === 'tag'): ?>
<!-- ═══════ TAG ═══════ -->
  <div class="mi">🏷 Tag: <b style="color:var(--ac)"><?=$hT?></b> — <?=number_format($tot)?> sites · page <?=$PG?>/<?=$pgs?></div>
  <?php if ($allTags): ?><div class="tl2" style="margin-bottom:18px"><?php foreach(array_slice($allTags,0,30,true) as $t=>$c):?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="tag"><input type="hidden" name="tag" value="<?=htmlspecialchars($t)?>"><button type="submit" class="tp <?=$t===$TAG?'on':''?>"><?=htmlspecialchars($t)?></button></form>
  <?php endforeach;?></div><?php endif; ?>
  <?php if ($err): ?><div style="color:var(--w)"><?=$err?></div>
  <?php elseif (!$res): ?><div class="empty">No sites with tag "<?=$hT?>"</div>
  <?php else: foreach ($res as $r): ?>
    <div class="card">
      <div class="ch">
        <span class="badge <?=($r['status']??'')==='online'?'b-on':'b-off'?>"><?=$r['status']?></span>
        <?php $cti=$r['cti_type']??''; if($cti): $cc=stripos($cti,'ransom')!==false?'b-ran':(stripos($cti,'forum')!==false?'b-for':'b-cti');?>
          <span class="badge <?=$cc?>"><?=htmlspecialchars($cti)?></span><?php endif;?>
        <?php if(!empty($sparks[$r['address']])):?><?=spark($sparks[$r['address']])?><?php endif;?>
      </div>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($r['address'])?>"><button type="submit" class="ct"><?=htmlspecialchars($r['title']?:'(untitled)')?></button></form>
      <div class="ca"><?=htmlspecialchars($r['address'])?></div>
      <?php if($r['meta_description']??''):?><div class="cd"><?=htmlspecialchars(mb_substr($r['meta_description'],0,250))?></div><?php endif;?>
      <div class="cm">
        <?php if($r['server_software']??''):?><span>🖥 <?=htmlspecialchars($r['server_software'])?></span><?php endif;?>
        <?php if($r['tech_stack']??''):?><span>⚙ <?=htmlspecialchars(mb_substr($r['tech_stack'],0,60))?></span><?php endif;?>
        <?php if($r['last_seen']??''):?><span>seen <?=substr($r['last_seen'],0,10)?></span><?php endif;?>
      </div>
      <?php $tags=json_decode($r['tags']?:'[]',true); if($tags):?>
      <div class="tl2"><?php foreach(array_slice($tags,0,6) as $t):?>
        <form method="post" style="display:inline"><input type="hidden" name="v" value="tag"><input type="hidden" name="tag" value="<?=htmlspecialchars($t)?>"><button type="submit" class="tp <?=$t===$TAG?'on':''?>"><?=htmlspecialchars($t)?></button></form>
      <?php endforeach;?></div><?php endif;?>
      <?php $_ri++;?><button class="rb" onclick="tr(<?=$_ri?>)">⚑ report</button>
      <div class="rx" id="rpt-<?=$_ri?>"><select><option value="csam">CSAM</option><option value="scam">Scam</option><option value="malware">Malware</option><option value="miscat">Miscategorized</option><option value="other">Other</option></select><textarea placeholder="Details…" maxlength="500"></textarea><button class="rs" onclick="sr(<?=$_ri?>,'<?=htmlspecialchars($r['address'], ENT_QUOTES)?>')">Submit</button></div>
    </div>
  <?php endforeach;
    if ($pgs > 1):?><div class="pager"><?php for($i=max(1,$PG-3);$i<=min($pgs,$PG+3);$i++):?>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="tag"><input type="hidden" name="tag" value="<?=$hT?>"><input type="hidden" name="p" value="<?=$i?>"><button type="submit" class="<?=$i===$PG?'on':''?>"><?=$i?></button></form>
    <?php endfor;?></div><?php endif; endif; ?>

<?php elseif ($V === 'detail' && $det): ?>
<!-- ═══════ SITE DETAIL ═══════ -->
  <div class="dt">
    <div class="ch" style="margin-bottom:8px">
      <span class="badge <?= $det['status']==='online'?'b-on':'b-off' ?>"><?= $det['status'] ?></span>
      <?php if($det['cti_type']):$cc=stripos($det['cti_type'],'ransom')!==false?'b-ran':(stripos($det['cti_type'],'forum')!==false?'b-for':'b-cti');?>
        <span class="badge <?=$cc?>"><?=htmlspecialchars($det['cti_type'])?></span><?php endif;?>
      <?php if($det['cti_name']):?><span style="font-size:.8rem;color:var(--w);font-weight:bold"><?=htmlspecialchars($det['cti_name'])?></span><?php endif;?>
    </div>
    <h2><?= htmlspecialchars($det['title'] ?: '(untitled)') ?></h2>
    <div class="da"><?= htmlspecialchars($det['address']) ?></div>
    <?php if($det['meta_description']):?><div class="cd" style="margin-bottom:12px"><?=htmlspecialchars(mb_substr($det['meta_description'],0,500))?></div><?php endif;?>
    <div class="dg">
      <span class="k">First seen</span><span class="v"><?=$det['first_seen']?></span>
      <span class="k">Last seen</span><span class="v"><?=$det['last_seen']?></span>
      <span class="k">Last checked</span><span class="v"><?=$det['last_checked']?:'never'?></span>
      <span class="k">Check count</span><span class="v"><?=$det['check_count']?></span>
      <span class="k">Offline streak</span><span class="v"><?=$det['offline_streak']?></span>
      <span class="k">Search hits</span><span class="v"><?=$det['hits']?></span>
      <?php if($det['server_software']):?><span class="k">Server</span><span class="v"><?=htmlspecialchars($det['server_software'])?></span><?php endif;?>
      <?php if($det['powered_by']):?><span class="k">Powered by</span><span class="v"><?=htmlspecialchars($det['powered_by'])?></span><?php endif;?>
      <?php if($det['meta_generator']):?><span class="k">Generator</span><span class="v"><?=htmlspecialchars($det['meta_generator'])?></span><?php endif;?>
      <?php if($det['tech_stack']):?><span class="k">Tech stack</span><span class="v"><?=htmlspecialchars($det['tech_stack'])?></span><?php endif;?>
      <?php if($det['page_language']):?><span class="k">Language</span><span class="v"><?=htmlspecialchars($det['page_language'])?></span><?php endif;?>
      <?php if($det['page_size']):?><span class="k">Page size</span><span class="v"><?=number_format($det['page_size'])?> bytes</span><?php endif;?>
      <?php if($det['response_time_ms']):?><span class="k">Response time</span><span class="v"><?=$det['response_time_ms']?> ms</span><?php endif;?>
      <?php if($det['link_count']):?><span class="k">Links</span><span class="v"><?=$det['link_count']?></span><?php endif;?>
      <?php if($det['form_count']):?><span class="k">Forms</span><span class="v"><?=$det['form_count']?></span><?php endif;?>
      <?php if($det['image_count']):?><span class="k">Images</span><span class="v"><?=$det['image_count']?></span><?php endif;?>
      <?php if($det['source']):?><span class="k">Source</span><span class="v"><?=htmlspecialchars($det['source'])?></span><?php endif;?>
      <?php if($det['og_title']):?><span class="k">OG Title</span><span class="v"><?=htmlspecialchars($det['og_title'])?></span><?php endif;?>
      <?php if($det['og_description']):?><span class="k">OG Description</span><span class="v"><?=htmlspecialchars(mb_substr($det['og_description'],0,200))?></span><?php endif;?>
    </div>
    <?php $dtags=json_decode($det['tags']?:'[]',true); if($dtags):?>
    <div class="tl2" style="margin-top:10px"><?php foreach($dtags as $t):?>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="tag"><input type="hidden" name="tag" value="<?=htmlspecialchars($t)?>"><button type="submit" class="tp"><?=htmlspecialchars($t)?></button></form>
    <?php endforeach;?></div><?php endif;?>
  </div>

  <!-- Uptime sparkline -->
  <?php $up = $det['_uptime'] ?? [];
  if ($up):
    $pts = array_map(fn($u) => (int)$u['alive'], $up);
    $onC = count(array_filter($pts));
    $pct = count($pts) ? round($onC / count($pts) * 100) : 0;
  ?>
  <div class="dt">
    <h3 style="font-size:.9rem;color:var(--ac);margin-bottom:4px">📊 30-Day Uptime</h3>
    <?= spark($pts, true) ?>
    <div class="up-st"><?= $onC ?>/<?= count($pts) ?> checks online (<?= $pct ?>% uptime) · <?= count($pts) ?> data points</div>
  </div>
  <?php endif; ?>

  <!-- Sub-pages -->
  <?php $subp = $det['_pages'] ?? [];
  if ($subp): ?>
  <div class="dt">
    <h3 style="font-size:.9rem;color:var(--ac);margin-bottom:8px">📄 Discovered Pages (<?= count($subp) ?>)</h3>
    <?php foreach (array_slice($subp, 0, 50) as $sp): ?>
    <div class="subp">
      <div style="color:var(--tor);font-size:.8rem;word-break:break-all"><?= htmlspecialchars(mb_substr($sp['page_url'],0,200)) ?></div>
      <?php if($sp['title']):?><div class="subp-t"><?=htmlspecialchars(mb_substr($sp['title'],0,100))?></div><?php endif;?>
      <div style="font-size:.65rem;color:var(--dm)">found <?=substr($sp['first_seen'],0,10)?> <?php if($sp['source']):?>· via <?=htmlspecialchars($sp['source'])?><?php endif;?></div>
    </div>
    <?php endforeach; ?>
    <?php if (count($subp) > 50): ?><div style="color:var(--dm);font-size:.75rem;margin-top:6px">…and <?= count($subp) - 50 ?> more</div><?php endif; ?>
  </div>
  <?php endif; ?>

  <!-- Report -->
  <?php $_ri++; ?>
  <button class="rb" onclick="tr(<?=$_ri?>)">⚑ Report this site</button>
  <div class="rx" id="rpt-<?=$_ri?>"><select><option value="csam">CSAM</option><option value="scam">Scam</option><option value="malware">Malware</option><option value="miscat">Miscategorized</option><option value="other">Other</option></select><textarea placeholder="Details…" maxlength="500"></textarea><button class="rs" onclick="sr(<?=$_ri?>,'<?=htmlspecialchars($det['address'], ENT_QUOTES)?>')">Submit</button></div>

  <!-- Canary -->
  <?php $can = $det['_canary'] ?? null; if ($can): ?>
  <div class="dt" style="margin-top:12px">
    <h3 style="font-size:.9rem;color:var(--ac);margin-bottom:6px">🐤 Warrant Canary</h3>
    <div class="ch"><span class="badge <?= $can['status']==='active'?'b-on':($can['status']==='changed'?'badge' : 'b-off') ?>" <?= $can['status']==='changed'?'style="background:var(--w);color:#fff"':'' ?>><?= $can['status'] ?></span>
      <span style="font-size:.7rem;color:var(--dm)">last seen <?= substr($can['last_seen'],0,16) ?></span></div>
    <?php if($can['preview']):?><div style="font-size:.75rem;color:var(--dm);margin-top:4px;white-space:pre-wrap;max-height:120px;overflow:auto;background:var(--s2);padding:8px;border-radius:4px"><?=htmlspecialchars(mb_substr($can['preview'],0,400))?></div><?php endif;?>
    <div style="font-size:.65rem;color:var(--dm);margin-top:4px">hash: <?=htmlspecialchars(substr($can['canary_hash'],0,16))?>… <?php if($can['last_hash']):?>· prev: <?=htmlspecialchars(substr($can['last_hash'],0,16))?>…<?php endif;?></div>
  </div>
  <?php endif; ?>

  <!-- Mirrors -->
  <?php $mir = $det['_mirrors'] ?? []; if ($mir): ?>
  <div class="dt" style="margin-top:12px">
    <h3 style="font-size:.9rem;color:var(--ac);margin-bottom:6px">🪞 Mirrors (<?= count($mir) ?> identical sites)</h3>
    <?php foreach ($mir as $m): ?>
    <div style="padding:4px 0;border-bottom:1px solid var(--bd)">
      <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($m['address'])?>">
        <button type="submit" class="ct" style="font-size:.85rem"><?=htmlspecialchars($m['title']?:$m['address'])?></button></form>
      <span class="badge <?=$m['status']==='online'?'b-on':'b-off'?>" style="margin-left:6px"><?=$m['status']?></span>
    </div>
    <?php endforeach; ?>
  </div>
  <?php endif; ?>

  <!-- Snapshots (Wayback) -->
  <?php $snaps = $det['_snapshots'] ?? []; if ($snaps): ?>
  <div class="dt" style="margin-top:12px">
    <h3 style="font-size:.9rem;color:var(--ac);margin-bottom:6px">⏪ Wayback Timeline (<?= count($snaps) ?> snapshots)</h3>
    <div style="display:flex;gap:3px;flex-wrap:wrap;margin-bottom:8px">
      <?php foreach ($snaps as $sn): $uniq = substr($sn['content_hash'],0,6); ?>
      <div title="<?=htmlspecialchars($sn['title'])?> — <?=$sn['captured_at']?>" style="width:12px;height:20px;border-radius:2px;background:var(--ac);opacity:0.6;cursor:help;position:relative" onmouseover="this.style.opacity=1" onmouseout="this.style.opacity=0.6"></div>
      <?php endforeach; ?>
    </div>
    <div style="max-height:200px;overflow:auto">
    <?php foreach ($snaps as $sn): ?>
    <div style="font-size:.75rem;padding:3px 0;border-bottom:1px solid var(--bd);display:flex;gap:10px;color:var(--dm)">
      <span style="color:var(--t);min-width:130px"><?= substr($sn['captured_at'],0,16) ?></span>
      <span style="flex:1"><?= htmlspecialchars($sn['title'] ?: '(untitled)') ?></span>
      <span><?= number_format($sn['page_size']) ?>b</span>
      <span style="font-family:monospace;font-size:.65rem;color:var(--ac)"><?= substr($sn['content_hash'],0,8) ?></span>
    </div>
    <?php endforeach; ?>
    </div>
  </div>
  <?php endif; ?>

  <!-- Related Sites -->
  <?php $rel = $det['_related'] ?? []; if ($rel): ?>
  <div class="dt" style="margin-top:12px">
    <h3 style="font-size:.9rem;color:var(--ac);margin-bottom:6px">🔗 Related Sites</h3>
    <?php foreach ($rel as $rs): ?>
    <div style="padding:4px 0;border-bottom:1px solid var(--bd)">
      <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($rs['address'])?>">
        <button type="submit" class="ct" style="font-size:.85rem"><?=htmlspecialchars($rs['title']?:$rs['address'])?></button></form>
      <span class="badge <?=$rs['status']==='online'?'b-on':'b-off'?>" style="margin-left:4px"><?=$rs['status']?></span>
      <?php if($rs['tech_stack']??''):?><span style="font-size:.65rem;color:var(--dm);margin-left:6px">⚙ <?=htmlspecialchars(mb_substr($rs['tech_stack'],0,40))?></span><?php endif;?>
    </div>
    <?php endforeach; ?>
  </div>
  <?php endif; ?>

<?php elseif ($V === 'detail'): ?>
  <div class="empty">Site not found.</div>

<?php elseif ($V === 'tech'): ?>
<!-- ═══════ TECH FINGERPRINT ═══════ -->
  <div class="mi">⚙ Tech Fingerprint Search — find sites by server stack</div>
  <form method="post" style="margin-bottom:16px"><input type="hidden" name="v" value="tech">
    <div class="sf" style="max-width:600px"><input type="text" name="tech" value="<?=htmlspecialchars($_R['tech']??'')?>" placeholder="nginx+PHP+WordPress (use + between terms)"><button type="submit">Search</button></div></form>
  <?php if (!empty($techTop ?? [])): ?>
  <div style="margin-bottom:16px"><span style="font-size:.75rem;color:var(--dm)">Top servers:</span>
    <div class="tl2" style="margin-top:4px"><?php foreach($techTop as $tt):?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="tech"><input type="hidden" name="tech" value="<?=htmlspecialchars($tt['server_software'])?>">
      <button type="submit" class="tp"><?=htmlspecialchars($tt['server_software'])?> <span style="opacity:.5"><?=$tt['c']?></span></button></form>
    <?php endforeach;?></div></div>
  <?php endif; ?>
  <?php if ($tot): ?><div class="mi"><?=number_format($tot)?> sites · page <?=$PG?>/<?=$pgs?></div><?php endif; ?>
  <?php foreach ($res as $r): ?>
    <div class="card"><div class="ch"><span class="badge <?=($r['status']??'')==='online'?'b-on':'b-off'?>"><?=$r['status']?></span>
      <?php if($r['server_software']??''):?><span style="font-size:.7rem;color:var(--cm);font-weight:bold">🖥 <?=htmlspecialchars($r['server_software'])?></span><?php endif;?>
      <?php if($r['tech_stack']??''):?><span style="font-size:.65rem;color:var(--dm)">⚙ <?=htmlspecialchars(mb_substr($r['tech_stack'],0,60))?></span><?php endif;?></div>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($r['address'])?>"><button type="submit" class="ct"><?=htmlspecialchars($r['title']?:'(untitled)')?></button></form>
    <div class="ca"><?=htmlspecialchars($r['address'])?></div></div>
  <?php endforeach; ?>
  <?php if ($pgs>1):?><div class="pager"><?php for($i=max(1,$PG-3);$i<=min($pgs,$PG+3);$i++):?><form method="post" style="display:inline"><input type="hidden" name="v" value="tech"><input type="hidden" name="tech" value="<?=htmlspecialchars($_R['tech']??'')?>"><input type="hidden" name="p" value="<?=$i?>"><button type="submit" class="<?=$i===$PG?'on':''?>"><?=$i?></button></form><?php endfor;?></div><?php endif;?>

<?php elseif ($V === 'lang'): ?>
<!-- ═══════ LANGUAGE BROWSE ═══════ -->
  <div class="mi">🌐 Browse by Language</div>
  <?php if (!empty($langTop ?? [])): ?>
  <div class="tl2" style="margin-bottom:16px"><?php foreach($langTop as $ll):?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="lang"><input type="hidden" name="lang" value="<?=htmlspecialchars($ll['page_language'])?>">
      <button type="submit" class="tp <?=($ll['page_language']===($LANG??''))?'on':''?>"><?=htmlspecialchars($ll['page_language'])?> <span style="opacity:.5"><?=$ll['c']?></span></button></form>
  <?php endforeach;?></div>
  <?php endif; ?>
  <?php if ($tot): ?><div class="mi"><?=number_format($tot)?> sites in "<?=htmlspecialchars($LANG)?>" · page <?=$PG?>/<?=$pgs?></div><?php endif; ?>
  <?php foreach ($res as $r): ?>
    <div class="card"><div class="ch"><span class="badge <?=($r['status']??'')==='online'?'b-on':'b-off'?>"><?=$r['status']?></span></div>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($r['address'])?>"><button type="submit" class="ct"><?=htmlspecialchars($r['title']?:'(untitled)')?></button></form>
    <div class="ca"><?=htmlspecialchars($r['address'])?></div>
    <?php if($r['meta_description']??''):?><div class="cd"><?=htmlspecialchars(mb_substr($r['meta_description'],0,180))?></div><?php endif;?></div>
  <?php endforeach; ?>
  <?php if ($pgs>1):?><div class="pager"><?php for($i=max(1,$PG-3);$i<=min($pgs,$PG+3);$i++):?><form method="post" style="display:inline"><input type="hidden" name="v" value="lang"><input type="hidden" name="lang" value="<?=htmlspecialchars($LANG)?>"><input type="hidden" name="p" value="<?=$i?>"><button type="submit" class="<?=$i===$PG?'on':''?>"><?=$i?></button></form><?php endfor;?></div><?php endif;?>

<?php elseif ($V === 'changes'): ?>
<!-- ═══════ CHANGES FEED ═══════ -->
  <?php $cType=$_R['ctype']??''; $cDays=$_R['cdays']??1; ?>
  <div class="mi">🔄 Change Detection Feed — what changed <?= $cDays > 1 ? "in the last $cDays days" : "today" ?></div>
  <div style="display:flex;gap:4px;margin-bottom:12px;flex-wrap:wrap">
    <?php $types=[''=>'All','came_online'=>'🟢 Online','went_offline'=>'🔴 Offline','title_changed'=>'📝 Titles','content_changed'=>'📄 Content','new_pages'=>'📑 Pages','canary_changed'=>'🐤 Canaries'];
    foreach($types as $tk=>$tl):?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="changes"><input type="hidden" name="ctype" value="<?=$tk?>"><input type="hidden" name="cdays" value="<?=$cDays?>">
      <button type="submit" class="tp <?=$tk===$cType?'on':''?>"><?=$tl?></button></form>
    <?php endforeach;?>
    <span style="color:var(--dm);font-size:.7rem;margin-left:8px">Days:</span>
    <?php foreach([1,3,7,30] as $d):?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="changes"><input type="hidden" name="ctype" value="<?=$cType?>"><input type="hidden" name="cdays" value="<?=$d?>">
      <button type="submit" class="tp <?=intval($cDays)===$d?'on':''?>"><?=$d?>d</button></form>
    <?php endforeach;?>
  </div>
  <?php if (!$res): ?><div class="empty">No changes detected <?= $cDays > 1 ? "in the last $cDays days" : "today" ?>.</div>
  <?php else: foreach ($res as $ch):
    $icon = ['title_changed'=>'📝','came_online'=>'🟢','went_offline'=>'🔴','new_pages'=>'📑','content_changed'=>'📄','canary_changed'=>'🐤'][$ch['change_type']]??'•'; ?>
    <div class="pc">
      <div class="ch"><span style="font-size:.9rem"><?=$icon?></span> <span class="badge b-cti" style="font-size:.6rem"><?=$ch['change_type']?></span>
        <span style="font-size:.65rem;color:var(--dm)"><?=substr($ch['detected_at'],0,16)?></span></div>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($ch['address'])?>">
        <button type="submit" class="ct" style="font-size:.85rem"><?=htmlspecialchars($ch['site_title']?:$ch['address'])?></button></form>
      <div style="font-size:.75rem;color:var(--dm);margin-top:2px">
        <?php if($ch['old_val']):?><span style="text-decoration:line-through;color:var(--w)"><?=htmlspecialchars(mb_substr($ch['old_val'],0,80))?></span> → <?php endif;?>
        <?php if($ch['new_val']):?><span style="color:var(--on)"><?=htmlspecialchars(mb_substr($ch['new_val'],0,80))?></span><?php endif;?>
      </div>
    </div>
  <?php endforeach; endif; ?>
  <?php if ($pgs > 1): ?><div class="pager"><?php for($i=max(1,$PG-3);$i<=min($pgs,$PG+3);$i++): ?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="changes"><input type="hidden" name="ctype" value="<?=htmlspecialchars($cType)?>"><input type="hidden" name="cdays" value="<?=$cDays?>"><input type="hidden" name="p" value="<?=$i?>"><button type="submit" class="<?=$i===$PG?'on':''?>"><?=$i?></button></form>
  <?php endfor; ?></div><?php endif; ?>

<?php elseif ($V === 'canaries'): ?>
<!-- ═══════ CANARY TRACKER ═══════ -->
  <div class="mi">🐤 Warrant Canary Tracker — <?=$tot?> canaries monitored</div>
  <?php if (!$res): ?><div class="empty">No canaries detected yet. The crawler looks for PGP blocks and "warrant canary" text.</div>
  <?php else: foreach ($res as $cn): ?>
    <div class="pc">
      <div class="ch">
        <span class="badge <?=$cn['status']==='active'?'b-on':($cn['status']==='changed'?'badge':'b-off')?>" <?=$cn['status']==='changed'?'style="background:var(--w);color:#fff"':''?>><?=$cn['status']?></span>
        <span style="font-size:.65rem;color:var(--dm)">last <?=substr($cn['last_seen'],0,10)?></span>
      </div>
      <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($cn['address'])?>">
        <button type="submit" class="ct" style="font-size:.85rem"><?=htmlspecialchars($cn['site_title']?:$cn['address'])?></button></form>
      <?php if($cn['preview']):?><div style="font-size:.7rem;color:var(--dm);margin-top:3px;max-height:60px;overflow:hidden"><?=htmlspecialchars(mb_substr($cn['preview'],0,200))?></div><?php endif;?>
      <div style="font-size:.6rem;color:var(--dm);margin-top:2px">hash: <?=substr($cn['canary_hash'],0,16)?>… <?php if($cn['last_hash']):?>prev: <?=substr($cn['last_hash'],0,16)?>…<?php endif;?></div>
    </div>
  <?php endforeach; endif; ?>
  <?php if ($pgs > 1): ?><div class="pager"><?php for($i=max(1,$PG-3);$i<=min($pgs,$PG+3);$i++): ?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="canaries"><input type="hidden" name="p" value="<?=$i?>"><button type="submit" class="<?=$i===$PG?'on':''?>"><?=$i?></button></form>
  <?php endfor; ?></div><?php endif; ?>

<?php elseif ($V === 'mirrors'): ?>
<!-- ═══════ MIRROR GROUPS ═══════ -->
  <div class="mi">🪞 Mirror Groups — <?=$tot?> groups of identical sites</div>
  <?php if (!$res): ?><div class="empty">No mirrors detected. Sites with identical content hashes will appear here.</div>
  <?php else: foreach ($res as $mg):
    $addrs = explode('|', $mg['addrs']);
    $titles = explode('|', $mg['titles']); ?>
    <div class="pc">
      <div class="ch"><span class="badge b-cti"><?=$mg['cnt']?> mirrors</span>
        <span style="font-size:.65rem;color:var(--dm)">hash: <?=substr($mg['content_hash'],0,12)?>…</span></div>
      <?php foreach (array_slice($addrs,0,8) as $i=>$ma): ?>
      <div style="padding:2px 0">
        <form method="post" style="display:inline"><input type="hidden" name="v" value="detail"><input type="hidden" name="addr" value="<?=htmlspecialchars($ma)?>">
          <button type="submit" style="background:none;border:none;color:var(--tor);font:inherit;font-size:.8rem;cursor:pointer;padding:0"><?=htmlspecialchars($titles[$i]??$ma)?></button></form>
      </div>
      <?php endforeach; ?>
      <?php if(count($addrs)>8):?><div style="font-size:.7rem;color:var(--dm)">…+<?=count($addrs)-8?> more</div><?php endif;?>
    </div>
  <?php endforeach; endif; ?>
  <?php if ($pgs > 1): ?><div class="pager"><?php for($i=max(1,$PG-3);$i<=min($pgs,$PG+3);$i++): ?>
    <form method="post" style="display:inline"><input type="hidden" name="v" value="mirrors"><input type="hidden" name="p" value="<?=$i?>"><button type="submit" class="<?=$i===$PG?'on':''?>"><?=$i?></button></form>
  <?php endfor; ?></div><?php endif; ?>

<?php elseif ($V === 'submit'): ?>
<!-- ═══════ SUBMIT A SITE ═══════ -->
  <div class="mi">➕ Submit a .onion site to be indexed</div>
  <div class="dt">
    <div style="font-size:.8rem;color:var(--dm);margin-bottom:12px">Submit your .onion address. It will be reviewed and crawled if it passes our filters. CSAM and illegal content will be rejected and blocklisted.</div>
    <div id="submit-form">
      <div style="margin-bottom:8px"><label style="font-size:.75rem;color:var(--dm)">Onion address (v3, 56 chars):</label><br>
        <input type="text" id="sub-addr" placeholder="abc123…xyz.onion" style="width:100%;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:8px;border-radius:4px;font-family:inherit;font-size:.85rem;margin-top:4px"></div>
      <div style="margin-bottom:8px"><label style="font-size:.75rem;color:var(--dm)">Description (optional):</label><br>
        <textarea id="sub-desc" maxlength="500" placeholder="What is this site?" style="width:100%;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:8px;border-radius:4px;font-family:inherit;font-size:.85rem;min-height:50px;resize:vertical;margin-top:4px"></textarea></div>
      <div style="margin-bottom:8px" id="captcha-box"><span style="font-size:.75rem;color:var(--dm)">Loading captcha…</span></div>
      <button onclick="doSubmit()" class="rs" style="background:var(--ac)">Submit Site</button>
      <div id="sub-msg" style="margin-top:8px;font-size:.8rem"></div>
    </div>
  </div>
  <script>
  var _captcha={token:'',a:0,b:0};
  (function loadCaptcha(){
    fetch('api/captcha.php').then(function(r){return r.json()}).then(function(d){
      _captcha={token:d.token,a:d.a,b:d.b};
      document.getElementById('captcha-box').innerHTML=
        '<label style="font-size:.75rem;color:var(--dm)">Captcha: What is '+d.a+' + '+d.b+'?</label><br>'+
        '<input type="number" id="sub-captcha" style="width:100px;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:8px;border-radius:4px;font-family:inherit;margin-top:4px">';
    });
  })();
  function doSubmit(){
    var a=document.getElementById('sub-addr').value.trim().toLowerCase(),
        d=document.getElementById('sub-desc').value,
        el=document.getElementById('sub-captcha'),
        c=el?parseInt(el.value):-1,
        m=document.getElementById('sub-msg');
    m.style.color='var(--dm)';m.textContent='Submitting…';
    fetch('api/submit.php',{method:'POST',headers:{'Content-Type':'application/json'},
      body:JSON.stringify({address:a,description:d,captcha_answer:c,captcha_token:_captcha.token,_csrf:_csrf})})
    .then(function(r){return r.json()}).then(function(d){
      m.style.color=d.error?'var(--w)':'var(--on)';m.textContent=d.message||d.error||d.status;
    }).catch(function(){m.style.color='var(--w)';m.textContent='Network error.';});
  }
  </script>

<?php elseif ($V === 'privacy'): ?>
<!-- ═══════ PRIVACY POLICY ═══════ -->
<article class="legal" style="font-size:.85rem;line-height:1.8;color:var(--dm)">
  <h2 style="color:var(--t);font-size:1.2rem;margin-bottom:4px">Privacy Policy</h2>
  <p style="font-size:.7rem;margin-bottom:20px"><time datetime="<?= date('Y-m-d') ?>">Last updated: <?= date('F j, Y') ?></time></p>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">What We Index</h3>
    <p>In Umbra indexes publicly accessible metadata from .onion hidden services on the Tor network. This includes page titles, server response headers, technology fingerprints, HTTP status codes, content hashes, and detected page language. We do not store, cache, proxy, or reproduce any actual page content from indexed sites.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">What We Collect From You</h3>
    <p><strong style="color:var(--t)">Search queries:</strong> Logged to improve search relevance and generate trending terms. Not tied to any personal identifier.</p>
    <p><strong style="color:var(--t)">IP addresses:</strong> Temporarily hashed (SHA-256, one-way) for rate limiting and abuse prevention. We do not store raw IP addresses. Hashes are used for rate-limit windows and cannot be reversed to identify you.</p>
    <p><strong style="color:var(--t)">Reports and submissions:</strong> When you report a site or submit a .onion address, we store the content of your submission and a SHA-256 hash of your IP for abuse prevention. No other identifying information is collected.</p>
    <p><strong style="color:var(--t)">Analytics:</strong> We use third-party analytics to understand aggregate traffic patterns. This may involve cookies — see our analytics provider's own privacy policy for details. Analytics data is used solely to improve the service.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Data Retention</h3>
    <p>Uptime check history: 60 days. Site change logs: 90 days. Metadata snapshots: newest 20 per site. Unapproved trending terms: 30 days. Rate-limit records: duration of the rate window only. Reports and submissions: retained until resolved by an administrator, then archived indefinitely for audit purposes.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Your Rights</h3>
    <p>Under applicable data protection law (including GDPR for EU residents), you have the right to request access to, correction of, or deletion of personal data we may hold about you. Since we collect minimal personal data (hashed IPs, no accounts), most requests will relate to the removal of indexed metadata. See our <a href="?v=removal" style="color:var(--ac)">Removal Request</a> page.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Content Filtering</h3>
    <p>We enforce the Ahmia CSAM (Child Sexual Abuse Material) blocklist on all indexed data. Sites matching known CSAM hashes are automatically blocked and never appear in results. Users can report sites for review using the report button on any search result. Reported sites are reviewed by an administrator and may be blocked or removed.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Third Parties</h3>
    <p>We do not sell, trade, or share personal data with third parties. Indexed metadata is publicly accessible through our search interface and API. Analytics data is processed by our analytics provider under their own privacy policy.</p>
  </section>

  <section>
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Contact</h3>
    <p>For privacy inquiries, data deletion requests, or any questions about this policy, use the <a href="?v=removal" style="color:var(--ac)">Removal Request</a> form or email: <strong style="color:var(--t)">privacy@inumbra.com</strong></p>
  </section>
</article>

<?php elseif ($V === 'terms'): ?>
<!-- ═══════ TERMS OF USE ═══════ -->
<article class="legal" style="font-size:.85rem;line-height:1.8;color:var(--dm)">
  <h2 style="color:var(--t);font-size:1.2rem;margin-bottom:4px">Terms of Use</h2>
  <p style="font-size:.7rem;margin-bottom:20px"><time datetime="<?= date('Y-m-d') ?>">Last updated: <?= date('F j, Y') ?></time></p>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Service Description</h3>
    <p>In Umbra ("the Service") is a metadata-only search engine that indexes publicly accessible information from .onion hidden services on the Tor network. The Service stores metadata — titles, server headers, technology fingerprints, status codes, content hashes, and page language — but does not store, cache, host, proxy, or reproduce any actual content from indexed sites.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Acceptable Use</h3>
    <p>You may use In Umbra for lawful purposes including security research, journalism, threat intelligence, academic research, and general information gathering. You may not use the Service to facilitate illegal activity, harass individuals, or circumvent the access controls of indexed sites. Automated access (scraping, bulk API calls) beyond reasonable research use is prohibited without prior permission.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">No Endorsement</h3>
    <p>Inclusion of a .onion address in our index does not constitute endorsement, recommendation, or approval of the content or services found at that address. In Umbra does not verify, vet, or vouch for any indexed site. The Service is a neutral index of publicly discoverable metadata.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Content Removal and Takedown</h3>
    <p>We honor legitimate removal requests from site operators who wish to have their .onion address delisted, and from individuals who have identified personal information in our index. Submit requests via our <a href="?v=removal" style="color:var(--ac)">Removal Request</a> page. We aim to process all requests within 72 hours.</p>
    <p><strong style="color:var(--t)">CSAM and illegal content:</strong> Sites matching the Ahmia CSAM blocklist are automatically filtered. Users can report sites using the report button on search results. Reported content is reviewed and may be permanently blocked.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">DMCA and Copyright</h3>
    <p>In Umbra indexes metadata (titles, headers, technology fingerprints) and does not reproduce copyrighted content. If you believe metadata in our index infringes your copyright, contact us at <strong style="color:var(--t)">dmca@inumbra.com</strong> with the specific address, a description of the copyrighted work, and your contact information. We will review and respond within 72 hours.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Disclaimer of Warranties</h3>
    <p>The Service is provided "as is" without warranty of any kind. We do not guarantee the accuracy, completeness, reliability, or timeliness of any indexed metadata. Site status, titles, and other metadata may be outdated or inaccurate. You use the information provided by In Umbra at your own risk.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Limitation of Liability</h3>
    <p>To the maximum extent permitted by law, In Umbra and its operators shall not be liable for any direct, indirect, incidental, special, consequential, or punitive damages arising from your use of or inability to use the Service, including any harm resulting from accessing indexed sites or relying on indexed metadata.</p>
  </section>

  <section style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Modifications</h3>
    <p>We reserve the right to modify these terms at any time. Changes take effect immediately upon posting. Your continued use of the Service constitutes acceptance of modified terms. The "Last updated" date at the top of this page reflects the most recent revision.</p>
  </section>

  <section>
    <h3 style="color:var(--ac);font-size:.95rem;margin-bottom:8px">Contact</h3>
    <p>For legal inquiries: <strong style="color:var(--t)">legal@inumbra.com</strong></p>
    <p>For DMCA notices: <strong style="color:var(--t)">dmca@inumbra.com</strong></p>
    <p>For removal requests: <a href="?v=removal" style="color:var(--ac)">Removal Request Form</a></p>
  </section>
</article>

<?php elseif ($V === 'removal'): ?>
<!-- ═══════ REMOVAL REQUEST ═══════ -->
<article class="legal" style="font-size:.85rem;line-height:1.8;color:var(--dm)">
  <h2 style="color:var(--t);font-size:1.2rem;margin-bottom:4px">Removal Request</h2>
  <p style="margin-bottom:16px">If you operate a .onion hidden service and want it removed from our index, or if you have identified personal information about yourself in our metadata, submit a removal request below. We aim to process all requests within 72 hours.</p>

  <div class="dt" style="margin-bottom:20px">
    <h3 style="color:var(--ac);font-size:.9rem;margin-bottom:12px">What can be removed</h3>
    <p><strong style="color:var(--t)">Site operators:</strong> You can request delisting of any .onion address you operate. The address and all associated metadata (title, status history, sub-pages, snapshots, canary records) will be removed from our index and blocked from future crawling.</p>
    <p style="margin-top:8px"><strong style="color:var(--t)">Personal information:</strong> If your name, alias, email address, or other personally identifying information appears in indexed metadata (e.g., in a page title or server header), you can request its removal under GDPR right to erasure or equivalent privacy law.</p>
    <p style="margin-top:8px"><strong style="color:var(--t)">Abuse reports:</strong> To report CSAM, scams, malware, or other abusive content, use the ⚑ report button on any search result instead. Abuse reports go through a separate review process.</p>
  </div>

  <div class="dt">
    <h3 style="color:var(--ac);font-size:.9rem;margin-bottom:12px">Submit a removal request</h3>
    <div id="removal-form">
      <div style="margin-bottom:10px">
        <label style="font-size:.75rem;color:var(--dm)">Request type:</label><br>
        <select id="rem-type" style="width:100%;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:8px;border-radius:4px;font-family:inherit;font-size:.85rem;margin-top:4px">
          <option value="delist">Delist a .onion address (site operator)</option>
          <option value="personal">Remove personal information (individual)</option>
          <option value="dmca">Copyright/DMCA takedown</option>
          <option value="other">Other</option>
        </select>
      </div>
      <div style="margin-bottom:10px">
        <label style="font-size:.75rem;color:var(--dm)">.onion address (if applicable):</label><br>
        <input type="text" id="rem-addr" placeholder="abc123…xyz.onion" style="width:100%;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:8px;border-radius:4px;font-family:inherit;font-size:.85rem;margin-top:4px">
      </div>
      <div style="margin-bottom:10px">
        <label style="font-size:.75rem;color:var(--dm)">Contact email (for follow-up — we won't share this):</label><br>
        <input type="email" id="rem-email" placeholder="you@example.com" style="width:100%;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:8px;border-radius:4px;font-family:inherit;font-size:.85rem;margin-top:4px">
      </div>
      <div style="margin-bottom:10px">
        <label style="font-size:.75rem;color:var(--dm)">Details (what should be removed and why):</label><br>
        <textarea id="rem-detail" maxlength="2000" placeholder="Describe what you'd like removed and your relationship to it (site operator, named individual, copyright holder, etc.)" style="width:100%;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:8px;border-radius:4px;font-family:inherit;font-size:.85rem;min-height:100px;resize:vertical;margin-top:4px"></textarea>
      </div>
      <div style="margin-bottom:10px" id="rem-captcha-box"><span style="font-size:.75rem;color:var(--dm)">Loading captcha…</span></div>
      <button onclick="doRemoval()" class="rs" style="background:var(--ac)">Submit Request</button>
      <div id="rem-msg" style="margin-top:8px;font-size:.8rem"></div>
    </div>
  </div>

  <div style="margin-top:20px;padding:12px;background:var(--s);border:1px solid var(--bd);border-radius:6px">
    <h3 style="color:var(--ac);font-size:.9rem;margin-bottom:8px">Alternative contact</h3>
    <p>You can also send removal requests directly to <strong style="color:var(--t)">removal@inumbra.com</strong> or <strong style="color:var(--t)">privacy@inumbra.com</strong>. Include the .onion address (if applicable), what you want removed, and your relationship to the content.</p>
  </div>
</article>

<script>
var _remCaptcha={token:'',a:0,b:0};
(function(){
  fetch('api/captcha.php').then(function(r){return r.json()}).then(function(d){
    _remCaptcha={token:d.token,a:d.a,b:d.b};
    document.getElementById('rem-captcha-box').innerHTML=
      '<label style="font-size:.75rem;color:var(--dm)">Captcha: What is '+d.a+' + '+d.b+'?</label><br>'+
      '<input type="number" id="rem-captcha" style="width:100px;background:var(--s);border:1px solid var(--bd);color:var(--t);padding:8px;border-radius:4px;font-family:inherit;margin-top:4px">';
  });
})();
function doRemoval(){
  var t=document.getElementById('rem-type').value,
      a=document.getElementById('rem-addr').value.trim(),
      e=document.getElementById('rem-email').value.trim(),
      d=document.getElementById('rem-detail').value,
      el=document.getElementById('rem-captcha'),
      c=el?parseInt(el.value):-1,
      m=document.getElementById('rem-msg');
  if(!d){m.style.color='var(--w)';m.textContent='Please describe what should be removed.';return;}
  if(t==='delist' && a && !/^[a-z2-7]{56}\.onion$/.test(a.toLowerCase())){m.style.color='var(--w)';m.textContent='Invalid .onion address (must be 56-character v3)';return;}
  m.style.color='var(--dm)';m.textContent='Submitting…';
  fetch('api/removal.php',{method:'POST',headers:{'Content-Type':'application/json'},
    body:JSON.stringify({type:t,address:a,email:e,detail:d,captcha_answer:c,captcha_token:_remCaptcha.token,_csrf:_csrf})})
  .then(function(r){return r.json()}).then(function(d){
    m.style.color=d.error?'var(--w)':'var(--on)';m.textContent=d.message||d.error||d.status;
  }).catch(function(){m.style.color='var(--w)';m.textContent='Network error.';});
}
</script>

<?php endif; ?>

</main>
</div>
<?php endif; ?>

<footer class="footer" role="contentinfo">
  <p><strong><?= $SN ?></strong> · inumbra.com · dark web meta-search engine for OSINT research</p>
  <p style="margin-top:6px">Metadata-only index · no content stored · CSAM-filtered
  · <a href="api/feed.php?type=sites" rel="noopener">📡 RSS Sites</a>
  · <a href="api/feed.php?type=pages" rel="noopener">📡 RSS Pages</a></p>
  <p style="margin-top:6px">
    <form method="post" style="display:inline"><input type="hidden" name="v" value="privacy"><button type="submit" style="background:none;border:none;color:var(--dm);font-family:inherit;font-size:.65rem;cursor:pointer;padding:0;letter-spacing:.1em">Privacy Policy</button></form>
    · <form method="post" style="display:inline"><input type="hidden" name="v" value="terms"><button type="submit" style="background:none;border:none;color:var(--dm);font-family:inherit;font-size:.65rem;cursor:pointer;padding:0;letter-spacing:.1em">Terms of Use</button></form>
    · <form method="post" style="display:inline"><input type="hidden" name="v" value="removal"><button type="submit" style="background:none;border:none;color:var(--dm);font-family:inherit;font-size:.65rem;cursor:pointer;padding:0;letter-spacing:.1em">Removal Request</button></form>
  </p>
  <p style="margin-top:6px"><time datetime="<?= date('Y-m-d') ?>">Last updated: <?= date('F j, Y') ?></time></p>
</footer>
</body>
</html>
